<?php

require_once('../includes/connection.php');
require_once('question_video_api.php');

header('Content-Type: application/json');

$institute_id = $_SESSION['institute_id'] ?? null;

function sanitizeInput($data) {
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

// Validate required fields
$requiredFields = ['question', 'category', 'level', 'criteria'];
foreach ($requiredFields as $field) {
    if (empty($_POST[$field])) {
        echo json_encode(["status" => "error", "message" => "$field is required"]);
        exit();
    }
}

// Assign variables from $_POST
$categoryCourseId = sanitizeInput($_POST['category']);
$level            = sanitizeInput($_POST['level']);
$criteria         = sanitizeInput($_POST['criteria']);
$type             = $_POST['type'] ?? null;
$media             = $_POST['media_select'] ?? null;
$question         = sanitizeInput($_POST['question']);
$videoMode         = trim($_POST['video_mode'] ?? '');
$videoUrl          = $_POST['video_url'] ?? null;
$video_title       = $_POST['video_title'] ?? null;
$marks              = $_POST['marks'] ?? null;
$duration           = $_POST['duration'] ?? null;
$yes_no              = $_POST['yes_no'] ?? null;
$true_false           = $_POST['true_false'] ?? null;

$createdBy        = $_SESSION['user_id'] ?? null;
$instituteId      = $_SESSION['institute_id'] ?? null;
$imageUrl         = null;

// Handle image upload
if (!empty($_FILES['image_upload']['name']) && $_FILES['image_upload']['error'] === UPLOAD_ERR_OK) {
    $file        = $_FILES['image_upload'];
    $fileName    = $file['name'];
    $fileTmpPath = $file['tmp_name'];
    $ext         = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
    $encrypted   = bin2hex(random_bytes(16)) . "." . $ext;

    // get institute folder
    $stmt = $pdo->prepare("SELECT folder_name FROM institute_master WHERE institute_id = :id");
    $stmt->execute([':id' => $institute_id]);
    $folder = $stmt->fetchColumn();
    if (!$folder) throw new Exception("Institute folder not found.");

    $uploadDir = "../institutes/$folder/question_images/";
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }

    $destPath = $uploadDir . $encrypted;

    if (!move_uploaded_file($fileTmpPath, $destPath)) {
        echo json_encode(["status" => "error", "message" => "Failed to upload image."]);
        exit();
    }

    // ✅ Save only filename in DB (consistent with update code)
    $imageUrl = $encrypted;
}


// Handle video upload
if (!empty($_FILES['video_upload']['name']) && $_FILES['video_upload']['error'] === UPLOAD_ERR_OK) {
    $videoFile = $_FILES['video_upload'];
    $response = uploadStudyMaterial($videoFile, $video_title, $categoryCourseId);

    if (!empty($response['success'])) {
        $videoUrl  = $response['video_link'];
    } else {
        echo json_encode(["status" => "error", "message" => $response['message'] ?? "Video upload failed"]);
        exit();
    }
}

// Normalize media flags
switch ($media) {
    case "Text":
        $videoUrl = $imageUrl = null;
        break;
    case "Image":
        $videoUrl = null;
        break;
    case "Video":
        $imageUrl = null;
        break;
    default:
        echo json_encode(["status" => "error", "message" => "Invalid media type"]);
        exit();
}

// Handle options and answer
$option1 = $option2 = $option3 = $option4 = null;
if (!in_array($type, ['true_false', 'yes_no'])) {
    $option1 = sanitizeInput($_POST['option_1'] ?? '');
    $option2 = sanitizeInput($_POST['option_2'] ?? '');
    $option3 = sanitizeInput($_POST['option_3'] ?? '');
    $option4 = sanitizeInput($_POST['option_4'] ?? '');
}

if ($type === "multiple_answers" && isset($_POST['multiple_options']) && is_array($_POST['multiple_options'])) {
    $correctAnswer = implode(",", array_map('trim', $_POST['multiple_options']));
} elseif ($type === "yes_no") {
    $correctAnswer = sanitizeInput($_POST['correct_answer'] ?? '');
} elseif ($type === "true_false") {
    $correctAnswer = sanitizeInput($_POST['correct_answer'] ?? '');
} else {
    $correctAnswer = sanitizeInput($_POST['correct_answer'] ?? '');
}

// Insert into DB
try {

    if ($media === 'Image' && !$imageUrl) {
        echo json_encode(["status" => "error", "message" => "Image upload failed or missing."]);
        exit();
    }

    if ($media === 'Video' && !$videoUrl) {
        echo json_encode(["status" => "error", "message" => "Video upload failed or missing."]);
        exit();
    }

    // ✅ Check for duplicate question in same category and institute
    $checkStmt = $pdo->prepare("SELECT COUNT(*) FROM question_master 
        WHERE question = :question AND category_course_id = :category_course_id AND institute_id = :institute_id");
    $checkStmt->execute([
        ':question' => $question,
        ':category_course_id' => $categoryCourseId,
        ':institute_id' => $instituteId
    ]);
    $existing = $checkStmt->fetchColumn();

    if ($existing > 0) {
        echo json_encode(["status" => "error", "message" => "Question already exists in this category."]);
        exit();
    }

    $stmt = $pdo->prepare("INSERT INTO question_master 
        (category_course_id, type, criteria, level, media, question, video_type, video_url, video_title, image_url, 
         option_1, option_2, option_3, option_4, correct_answer, marks, duration, created_by, institute_id, created_date)
        VALUES 
        (:category_course_id, :type, :criteria, :level, :media, :question, :vtype, :video_url, :video_title, :image_url, 
         :option_1, :option_2, :option_3, :option_4, :correct_answer, :marks, :duration, :created_by, :institute_id, NOW())");

    $stmt->execute([
        ':category_course_id' => $categoryCourseId,
        ':type'               => $type,
        ':criteria'           => $criteria,
        ':level'              => $level,
        ':media'               => $media,
        ':question'            => $question,
        ':vtype'            =>   $videoMode,
        ':video_url'        => $videoUrl,
        ':video_title'      => $video_title,
        ':image_url'        => $imageUrl,
        ':option_1'         => $option1,
        ':option_2'         => $option2,
        ':option_3'         => $option3,
        ':option_4'         => $option4,
        ':correct_answer'   => $correctAnswer,
        ':marks'            => $marks,
        ':duration'         => $duration,
        ':created_by'       => $createdBy,
        ':institute_id'     => $instituteId
    ]);

    echo json_encode(["status" => "success", "message" => "Question added successfully!"]);

} catch (PDOException $e) {
    echo json_encode(["status" => "error", "message" => "Database error: " . $e->getMessage()]);
}

?>
