<?php

// Crucial for debugging during development:
// These lines will display ALL PHP errors directly. REMOVE THEM IN PRODUCTION.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Ensure no accidental whitespace before the header
ob_start(); // Start output buffering

header('Content-Type: application/json'); // Set content type for JSON response

require_once('../includes/connection.php');
require('../vendor/autoload.php');

use Vimeo\Vimeo;

// Vimeo credentials - ideally move to environment variables or a secure config
// For production, always use environment variables or a secure configuration file.
// Hardcoding credentials is risky in production.
define('VIMEO_CLIENT_ID', '5aa41d66ac6c8c8f7ac864af4455b38e58124df5');
define('VIMEO_CLIENT_SECRET', 'f9VSklt+8WdIEFjYZAR5Q+gdo6z+ZgJWPpcCeuQWsEpOG8ksBRHLJlnRgl0mJ6QyJv6Khcf5tiP4AplxYCpoeTlKisXZAUz5LombPknL7wTfpSmbdtupJGW/7OBM5');
define('VIMEO_ACCESS_TOKEN', '59049690f9eb19c23cd3aaa17a2c759e');

// --- Input Validation ---
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
    exit;
}

$requiredFields = ['module_id', 'submodule_name', 'draft_id', 'vidtitlesub'];
foreach ($requiredFields as $field) {
    if (empty($_POST[$field])) {
        http_response_code(400); // Bad Request
        echo json_encode(['success' => false, 'message' => "Missing required field: $field."]);
        exit;
    }
}

if (!isset($_FILES['study_files_sub']) || $_FILES['study_files_sub']['error'] !== UPLOAD_ERR_OK) {
    http_response_code(400); // Bad Request
    echo json_encode(['success' => false, 'message' => 'No file uploaded or an upload error occurred.']);
    exit;
}

// --- Sanitize and Assign Input Variables ---
$moduleId        = $_POST['module_id'];
$submoduleName   = trim($_POST['submodule_name']);
$vidtitle        = trim($_POST['vidtitlesub']);
$courseDraftId   = $_POST['draft_id'];
$fileTmp         = $_FILES['study_files_sub']['tmp_name'];
$fileName        = $_FILES['study_files_sub']['name'];
$fileExt         = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

// MIME type validation for enhanced security and reliability
$finfo    = finfo_open(FILEINFO_MIME_TYPE);
$fileType = finfo_file($finfo, $fileTmp);
finfo_close($finfo);

$allowedVideoTypes = ['video/mp4', 'video/avi', 'video/quicktime', 'video/x-matroska', 'video/webm'];
$allowedDocTypes   = ['application/pdf', 'image/jpeg', 'image/png', 'image/gif', 'image/webp']; // Added webp

try {
    // 1. Check if the submodule already exists or insert it into `submodule_master`
    $stmt = $pdo->prepare("SELECT submodule_id, study_material FROM submodule_master WHERE submodule_name = :submodule_name");
    $stmt->execute([':submodule_name' => $submoduleName]);
    $submodule = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($submodule) {
        $submoduleId = $submodule['submodule_id'];
        $existingMaterial = $submodule['study_material']; // Existing study material IDs for this submodule
    } else {
        // Submodule does not exist, so insert it
        $stmt = $pdo->prepare("INSERT INTO submodule_master (submodule_name) VALUES (:submodule_name)");
        $stmt->execute([':submodule_name' => $submoduleName]);
        $submoduleId = $pdo->lastInsertId();
        $existingMaterial = ''; // New submodule, no existing materials yet
    }

    // 2. Update `course_draft_modules` to link the submodule
    // This section ensures the 'sub_modules' column is a comma-separated list of submodule IDs
    $stmt = $pdo->prepare("SELECT sub_modules FROM course_draft_modules WHERE course_draft_id = :draft_id AND module_id = :module_id");
    $stmt->execute([':draft_id' => $courseDraftId, ':module_id' => $moduleId]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    $currentSubmodules = '';
    if ($result) {
        $currentSubmodules = $result['sub_modules'];
    }

    // Convert existing comma-separated string to an array, filter out empty elements,
    // add the new submodule ID if it's not already present, then convert back.
    $submoduleArray = array_filter(explode(',', $currentSubmodules));
    if (!in_array($submoduleId, $submoduleArray)) {
        $submoduleArray[] = $submoduleId;
    }
    $updatedSubmodules = implode(',', $submoduleArray);

    if ($result) {
        // If an entry exists, update the `sub_modules` column
        $stmt = $pdo->prepare("UPDATE course_draft_modules SET sub_modules = :updated_submodules WHERE course_draft_id = :draft_id AND module_id = :module_id");
        $stmt->execute([
            ':updated_submodules' => $updatedSubmodules,
            ':draft_id' => $courseDraftId,
            ':module_id' => $moduleId
        ]);
    } else {
        // If no entry exists, insert a new row
        $stmt = $pdo->prepare("INSERT INTO course_draft_modules (course_draft_id, module_id, sub_modules) VALUES (:draft_id, :module_id, :sub_modules)");
        $stmt->execute([
            ':draft_id' => $courseDraftId,
            ':module_id' => $moduleId,
            ':sub_modules' => $updatedSubmodules
        ]);
    }

    // --- 3. Handle File Upload (Vimeo for video, local for docs/images) ---
    $statusMsg = '';
    $studyId = null; // Will store the ID of the new study material entry

    if (in_array($fileType, $allowedVideoTypes)) {
        // Video upload to Vimeo
        $client = new Vimeo(VIMEO_CLIENT_ID, VIMEO_CLIENT_SECRET, VIMEO_ACCESS_TOKEN);
        $uri = $client->upload($fileTmp, [
            "name" => $vidtitle,
            "description" => "Draft ID: $courseDraftId, Module ID: $moduleId, Submodule Name: $submoduleName",
            "privacy" => [ // Recommended privacy settings for uploads
                "view" => "disable", // Or "unlisted", "password" depending on your needs
                "embed" => "private" // Or "public"
            ]
        ]);

        $response = $client->request($uri . '?fields=transcode.status,link');
        $transcodeStatus = $response['body']['transcode']['status'] ?? 'unknown';
        $videoLink = $response['body']['link'] ?? '';
        $videoId = '';

        if (!empty($videoLink)) {
            // Extract video ID from Vimeo link (e.g., "https://vimeo.com/123456789" -> "123456789")
            $path = parse_url($videoLink, PHP_URL_PATH);
            $videoId = basename($path);
        }

        if (empty($videoId)) {
            throw new Exception("Failed to retrieve video ID from Vimeo response after upload.");
        }

        // Insert Vimeo video details into `study_material` table
        $stmt = $pdo->prepare("INSERT INTO study_material (vimeo_link, title) VALUES (:vimeo_link, :title)");
        $stmt->execute([':vimeo_link' => $videoId, ':title' => $vidtitle]);
        $studyId = $pdo->lastInsertId();

        $statusMsg = ($transcodeStatus === 'complete') ? 'Video uploaded and transcoded successfully.' :
                     (($transcodeStatus === 'in_progress') ? 'Video uploaded. Transcoding in progress.' :
                     'Video upload encountered a transcoding issue. Please check Vimeo.');

    } elseif (in_array($fileType, $allowedDocTypes)) {
        // Document/Image upload to local folder
        $uploadDir = '../study/'; // Ensure this directory has write permissions
        if (!file_exists($uploadDir)) {
            mkdir($uploadDir, 0755, true); // Create directory if it doesn't exist
        }

        // Generate a unique file name to prevent conflicts (e.g., if multiple users upload 'image.png')
        $uniqueFileName = uniqid() . '_' . preg_replace('/[^a-zA-Z0-9_\.-]/', '_', $fileName);
        $destination = $uploadDir . $uniqueFileName;

        if (!move_uploaded_file($fileTmp, $destination)) {
            throw new Exception("Failed to save uploaded file to server.");
        }

        // Insert local file details into `study_material` table
        $stmt = $pdo->prepare("INSERT INTO study_material (actual_file_name, title) VALUES (:actual_file_name, :title)");
        $stmt->execute([':actual_file_name' => $uniqueFileName, ':title' => $vidtitle]);
        $studyId = $pdo->lastInsertId();

        $statusMsg = 'Document uploaded successfully.';
    } else {
        // File type not allowed
        http_response_code(415); // Unsupported Media Type
        echo json_encode(['success' => false, 'message' => 'Unsupported file type. Allowed: MP4, AVI, MOV, MKV, WebM videos; PDF, JPG, PNG, GIF, WebP images.']);
        exit;
    }

    // 4. Update the `study_material` field in `submodule_master` to link the new content ID
    // This maintains a comma-separated list of study material IDs for the specific submodule
    $materialsArray = array_filter(explode(',', $existingMaterial)); // Filter out empty strings from existing list
    if ($studyId !== null && !in_array($studyId, $materialsArray)) { // Only add if it's a valid ID and not already present
        $materialsArray[] = $studyId;
    }
    $newStudyMaterial = implode(',', $materialsArray);

    $stmt = $pdo->prepare("UPDATE submodule_master SET study_material = :material WHERE submodule_id = :id");
    $stmt->execute([':material' => $newStudyMaterial, ':id' => $submoduleId]);

    // --- Final Success Response ---
    http_response_code(200); // OK
    echo json_encode(['success' => true, 'message' => $statusMsg, 'content_id' => $studyId, 'submodule_id' => $submoduleId]);

} catch (Exception $e) {
    // Catch any exceptions during the process and return an high-level error to the client
    http_response_code(500); // Internal Server Error
    error_log("Submodule content upload error: " . $e->getMessage() . " on line " . $e->getLine() . " in " . $e->getFile()); // Log detailed error for debugging
    echo json_encode(['success' => false, 'message' => 'A server error occurred during upload. Please try again.']);
}

// Ensure nothing else is outputted after the JSON
ob_end_flush();