<?php
header('Content-Type: application/json');
require_once('../includes/connection.php'); // Adjust path as needed

// --- Input Validation and Sanitization ---
$submoduleId = isset($_POST['submodule_id']) ? intval($_POST['submodule_id']) : 0; // Use 'submodule_id' as per frontend
$contentId = isset($_POST['content_id']) ? trim($_POST['content_id']) : '';

if (!$submoduleId || $contentId === '') {
    http_response_code(400); // Bad Request
    echo json_encode(['success' => false, 'message' => 'Invalid submodule ID or content ID provided.']);
    exit;
}

try {
    // 1. Fetch current study_material for the submodule from `submodule_master`
    // Ensure the table name is correct: `submodule_master`
    $sqlFetch = "SELECT study_material FROM submodule_master WHERE submodule_id = :submodule_id";
    $stmtFetch = $pdo->prepare($sqlFetch);
    $stmtFetch->execute([':submodule_id' => $submoduleId]);
    $row = $stmtFetch->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        http_response_code(404); // Not Found
        echo json_encode(['success' => false, 'message' => 'Submodule not found.']);
        exit;
    }

    $currentMaterials = $row['study_material'];
    $materialsArray = array_map('trim', explode(',', $currentMaterials)); // Trim each ID

    // Remove the contentId from the array. Using array_filter ensures elements are re-indexed numerically.
    $updatedMaterialsArray = array_filter($materialsArray, function($val) use ($contentId) {
        return $val !== $contentId; // Strict comparison after trimming
    });

    // Rebuild the study_material string. If empty, implode returns an empty string.
    $newStudyMaterial = implode(',', $updatedMaterialsArray);

    // 2. Update the `submodule_master` table with the new study_material list
    $sqlUpdate = "UPDATE submodule_master SET study_material = :study_material WHERE submodule_id = :submodule_id";
    $stmtUpdate = $pdo->prepare($sqlUpdate);
    $stmtUpdate->execute([
        ':study_material' => $newStudyMaterial,
        ':submodule_id' => $submoduleId
    ]);

    // Optional: Delete the actual study material record if no other submodule uses it
    // This requires checking other submodules and might be a complex decision.
    // For now, we're just delinking. If you want to delete the file/Vimeo video,
    // you'd add logic here to check `study_material` table, fetch `actual_file_name` or `vimeo_link`,
    // and then delete the file or Vimeo video.

    http_response_code(200); // OK
    echo json_encode([
        'success' => true,
        'message' => 'Content successfully removed from submodule.',
        'submodule_id' => $submoduleId,
        'removed_content_id' => $contentId,
        'updated_study_material_list' => $newStudyMaterial // Provide the new list for verification
    ]);

} catch (PDOException $e) {
    http_response_code(500); // Internal Server Error
    error_log("Database error in remove_study_material_api.php: " . $e->getMessage()); // Log error for debugging
    echo json_encode(['success' => false, 'message' => 'A database error occurred. Please try again.']);
} catch (Exception $e) {
    http_response_code(500); // Internal Server Error
    error_log("General error in remove_study_material_api.php: " . $e->getMessage()); // Log error for debugging
    echo json_encode(['success' => false, 'message' => 'A server error occurred. Please try again.']);
}

?>