<?php
require_once('../includes/connection.php');
header('Content-Type: application/json');

// Handle both JSON and multipart/form-data
if ($_SERVER['CONTENT_TYPE'] === 'application/json') {
    $data = json_decode(file_get_contents("php://input"), true);
} else {
    $data = $_POST;
}

if (
    isset($data['institute_name'], $data['phone_no'], $data['email_id'], $data['password']) &&
    !empty($data['institute_name']) && !empty($data['phone_no']) &&
    !empty($data['email_id']) && !empty($data['password'])
) {
    $instituteName = trim($data['institute_name']);
    $phoneNo = trim($data['phone_no']);
    $emailId = trim($data['email_id']);
    $password = base64_encode(trim($data['password']));
    $city = trim($data['city']);
    $state = trim($data['state']);
    $country = trim($data['country']);
    $pincode = trim($data['pincode']);
    $blockInstitute = $data['block_institute'];
    $instituteId = base64_decode($data['institute_id']);

    $selectedCoursesArray = isset($data['selected_courses']) ? $data['selected_courses'] : [];
    
    // Fetch old data
    $stmtOld = $pdo->prepare("SELECT email_id, password, user_id FROM institute_master WHERE institute_id = :id");
    $stmtOld->execute([':id' => $instituteId]);
    $oldInstitute = $stmtOld->fetch(PDO::FETCH_ASSOC);

    if (!$oldInstitute) {
        echo json_encode(["status" => "error", "message" => "Institute not found."]);
        exit();
    }

    // Check phone number uniqueness
    $checkPhoneStmt = $pdo->prepare("SELECT COUNT(*) FROM user_master WHERE phone_no = :phone_no AND user_id != :user_id");
    $checkPhoneStmt->execute([':phone_no' => $phoneNo, ':user_id' => $oldInstitute['user_id']]);
    if ($checkPhoneStmt->fetchColumn() > 0) {
        echo json_encode(["status" => "error", "message" => "Phone number already exists."]);
        exit();
    }

    // Check email uniqueness
    $checkEmailStmt = $pdo->prepare("SELECT COUNT(*) FROM user_master WHERE email_id = :email_id AND user_id != :user_id");
    $checkEmailStmt->execute([':email_id' => $emailId, ':user_id' => $oldInstitute['user_id']]);
    if ($checkEmailStmt->fetchColumn() > 0) {
        echo json_encode(["status" => "error", "message" => "Email ID already exists."]);
        exit();
    }

    // If we reach here, the duplicates checks have passed, and we can continue
    // Get folder_name from DB
    $getFolderStmt = $pdo->prepare("SELECT folder_name, institute_img FROM institute_master WHERE institute_id = :id");
    $getFolderStmt->execute([':id' => $instituteId]);
    $folderRow = $getFolderStmt->fetch(PDO::FETCH_ASSOC);

    if (!$folderRow || empty($folderRow['folder_name'])) {
        echo json_encode(["status" => "error", "message" => "Folder path not found for the institute."]);
        exit();
    }

    $folderName = $folderRow['folder_name'];
    $existingImg = $folderRow['institute_img'];
    $uploadDir = '../institutes/' . $folderName . '/institute_images/';

    // Ensure directory exists
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
        file_put_contents($uploadDir . 'index.html', '');
    }

    // Handle image upload
    $instituteImg = null;
    if (isset($_FILES['institute_img']) && $_FILES['institute_img']['error'] === UPLOAD_ERR_OK) {
        $fileExt = strtolower(pathinfo($_FILES['institute_img']['name'], PATHINFO_EXTENSION));
        $allowedTypes = ['jpg', 'jpeg', 'png', 'gif'];

        if (in_array($fileExt, $allowedTypes)) {
            $fileName = uniqid('institute_', true) . '.' . $fileExt;
            $targetFilePath = $uploadDir . $fileName;

            if (move_uploaded_file($_FILES['institute_img']['tmp_name'], $targetFilePath)) {
                // Optionally delete old image
                if (!empty($existingImg) && file_exists($uploadDir . $existingImg)) {
                    unlink($uploadDir . $existingImg);
                }
                $instituteImg = $fileName;
            } else {
                echo json_encode(["status" => "error", "message" => "Failed to move uploaded image."]);
                exit();
            }
        } else {
            echo json_encode(["status" => "error", "message" => "Invalid image file type."]);
            exit();
        }
    }
} else {
    echo json_encode(["status" => "error", "message" => "Institute name, phone number, and email are required"]);
    exit();
}

try {
    $pdo->beginTransaction();

    // Update institute_master (removed courses field)
    $updateInstituteSql = "UPDATE institute_master SET
        institute_name = :institute_name,
        phone_no = :phone_no,
        email_id = :email_id,
        password = :password,
        city = :city,
        state = :state,
        country = :country,
        pincode = :pincode,
        is_block = :block_institute,
        institute_img = COALESCE(NULLIF(:institute_img, ''), institute_img)
        WHERE institute_id = :institute_id";

    $pdo->prepare($updateInstituteSql)->execute([
        ':institute_name' => $instituteName,
        ':phone_no' => $phoneNo,
        ':email_id' => $emailId,
        ':password' => $password,
        ':city' => $city,
        ':state' => $state,
        ':country' => $country,
        ':pincode' => $pincode,
        ':block_institute' => $blockInstitute,
        ':institute_img' => $instituteImg,
        ':institute_id' => $instituteId
    ]);

    // Update user_master
    $pdo->prepare("UPDATE user_master SET
        email_id = :email_id,
        phone_no = :phone_no,
        password = :password,
        user_name = :user_name,
        is_block = :block
        WHERE user_id = :user_id")->execute([
        ':email_id' => $emailId,
        ':phone_no' => $phoneNo,
        ':password' => $password,
        ':user_name' => $instituteName,
        ':block' => $blockInstitute,
        ':user_id' => $oldInstitute['user_id']
    ]);

    // Delete and re-insert contact persons
    $pdo->prepare("DELETE FROM institute_contact_person WHERE institute_id = :institute_id")
        ->execute([':institute_id' => $instituteId]);

    if (!empty($data['contact_names']) && is_array($data['contact_names'])) {
        $insertContactSql = "INSERT INTO institute_contact_person (institute_id, contact_name, mobile_no, email_id)
                             VALUES (:institute_id, :contact_name, :mobile_no, :email_id)";
        $insertStmt = $pdo->prepare($insertContactSql);

        for ($i = 0; $i < count($data['contact_names']); $i++) {
            $name = trim($data['contact_names'][$i]);
            $phone = trim($data['contact_phones'][$i]);
            $email = trim($data['contact_emails'][$i]);

            if ($name || $phone || $email) {
                $insertStmt->execute([
                    ':institute_id' => $instituteId,
                    ':contact_name' => $name,
                    ':mobile_no' => $phone,
                    ':email_id' => $email
                ]);
            }
        }
    }

    // Delete old course mappings
    $pdo->prepare("DELETE FROM institute_courses WHERE institute_id = :institute_id")
        ->execute([':institute_id' => $instituteId]);

    // Insert new course mappings
    if (!empty($selectedCoursesArray)) {
        $insertCourseStmt = $pdo->prepare("INSERT INTO institute_courses (institute_id, course_id) VALUES (:institute_id, :course_id)");
        foreach ($selectedCoursesArray as $courseId) {
            $courseId = trim($courseId);
            if ($courseId !== '') {
                $insertCourseStmt->execute([
                    ':institute_id' => $instituteId,
                    ':course_id' => $courseId
                ]);
            }
        }
    }

    $pdo->commit();
    echo json_encode(["status" => "success", "message" => "Institute updated successfully!"]);
} catch (PDOException $e) {
    $pdo->rollBack();
    error_log("Edit Institute API Error: " . $e->getMessage());
    echo json_encode(["status" => "error", "message" => "An error occurred while updating the institute."]);
}
?>