<?php
header('Content-Type: application/json');
session_start(); // Assuming session is started and institute_id is available

require_once('../includes/connection.php');

// Ensure error reporting is configured for development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Input validation for draft_id
if (!isset($_GET['draft_id']) || empty($_GET['draft_id'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Missing or invalid draft_id.']);
    exit;
}

$draft_id = intval($_GET['draft_id']);

try {
    // 1. Fetch modules and their linked submodules from course_draft_modules
    //    along with the module_name from module_master.
    $sql = "
        SELECT
            cdm.draft_module_id,
            cdm.course_draft_id,
            cdm.module_id,
            mm.module_name,
            cdm.sub_modules AS linked_submodule_ids_csv
        FROM
            course_draft_modules cdm
        JOIN
            module_master mm ON cdm.module_id = mm.module_id
        WHERE
            cdm.course_draft_id = :draft_id
        ORDER BY
            cdm.draft_module_id ASC;
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':draft_id', $draft_id, PDO::PARAM_INT);
    $stmt->execute();
    $modulesData = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $finalOutput = [];

    // 2. Iterate through each module and fetch details for its linked submodules and their study materials
    foreach ($modulesData as $moduleEntry) {
        $submoduleEntry = [
            'draft_module_id' => $moduleEntry['draft_module_id'],
            'course_draft_id' => $moduleEntry['course_draft_id'],
            'module_id'       => $moduleEntry['module_id'],
            'module_name'     => $moduleEntry['module_name'],
            'submodules'      => [] // This will hold an array of submodule objects
        ];

        $linkedSubmoduleIdsCsv = $moduleEntry['linked_submodule_ids_csv'];

        if (!empty($linkedSubmoduleIdsCsv)) {
            $submoduleIds = array_map('intval', array_filter(explode(',', $linkedSubmoduleIdsCsv)));

            if (!empty($submoduleIds)) {
                // Generate placeholders for IN clause
                $inPlaceholders = implode(',', array_fill(0, count($submoduleIds), '?'));
                // Generate placeholders for FIELD clause
                $fieldPlaceholders = implode(',', array_fill(0, count($submoduleIds), '?'));

                $sqlSubmodules = "
                    SELECT
                        sm.submodule_id,
                        sm.submodule_name,
                        sm.study_material AS linked_study_material_ids_csv
                    FROM
                        submodule_master sm
                    WHERE
                        sm.submodule_id IN ($inPlaceholders)
                    ORDER BY FIELD(sm.submodule_id, $fieldPlaceholders); -- Maintain order of IDs
                ";

                $stmtSubmodules = $pdo->prepare($sqlSubmodules);
                // The parameters for execute are the IDs for IN, followed by the IDs for FIELD
                $stmtSubmodules->execute(array_merge($submoduleIds, $submoduleIds));
                $submodulesForModule = $stmtSubmodules->fetchAll(PDO::FETCH_ASSOC);

                foreach ($submodulesForModule as $submodule) {
                    $studyMaterials = [];
                    $linkedStudyMaterialIdsCsv = $submodule['linked_study_material_ids_csv'];

                    if (!empty($linkedStudyMaterialIdsCsv)) {
                        $materialIds = array_map('intval', array_filter(explode(',', $linkedStudyMaterialIdsCsv)));

                        if (!empty($materialIds)) {
                            // Generate placeholders for IN clause
                            $inMaterialPlaceholders = implode(',', array_fill(0, count($materialIds), '?'));
                            // Generate placeholders for FIELD clause
                            $fieldMaterialPlaceholders = implode(',', array_fill(0, count($materialIds), '?'));

                            $sqlStudyMaterial = "
                                SELECT
                                    study_material_id,
                                    title,
                                    actual_file_name,
                                    vimeo_link
                                FROM
                                    study_material
                                WHERE
                                    study_material_id IN ($inMaterialPlaceholders)
                                ORDER BY FIELD(study_material_id, $fieldMaterialPlaceholders); -- Maintain order
                            ";

                            $stmtStudyMaterial = $pdo->prepare($sqlStudyMaterial);
                            // The parameters for execute are the IDs for IN, followed by the IDs for FIELD
                            $stmtStudyMaterial->execute(array_merge($materialIds, $materialIds));
                            $studyMaterials = $stmtStudyMaterial->fetchAll(PDO::FETCH_ASSOC);
                        }
                    }
                    $submodule['contents'] = $studyMaterials; // Rename to 'contents' for clarity
                    unset($submodule['linked_study_material_ids_csv']); // Remove internal CSV field
                    $submoduleEntry['submodules'][] = $submodule;
                }
            }
        }
        $finalOutput[] = $submoduleEntry;
    }

    http_response_code(200);
    echo json_encode(['success' => true, 'modules' => $finalOutput]);

} catch (PDOException $e) {
    http_response_code(500);
    error_log("Database error in get_modules_and_submodules_by_draft.php: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(500);
    error_log("General error in get_modules_and_submodules_by_draft.php: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'An unexpected server error occurred.']);
}

// Flush the output buffer
ob_end_flush();
?>