<?php
header('Content-Type: application/json');
require_once('../includes/connection.php');

if (!isset($_GET['draft_id'])) {
    echo json_encode(['success' => false, 'message' => 'draft_id missing.']);
    exit;
}

$draft_id = intval($_GET['draft_id']);

try {
    // 1. Fetch module-submodule links from course_draft_modules
    // We'll fetch the comma-separated submodule_ids here.
    $stmt = $pdo->prepare("
        SELECT
            cdm.draft_module_id,
            cdm.course_draft_id,
            cdm.module_id,
            mm.module_name,
            cdm.sub_modules AS submodule_ids_csv -- Renamed to reflect it's CSV
        FROM
            course_draft_modules cdm
        JOIN
            module_master mm ON cdm.module_id = mm.module_id
        WHERE
            cdm.course_draft_id = :draft_id
        ORDER BY
            cdm.draft_module_id ASC;
    ");
    $stmt->execute(['draft_id' => $draft_id]);
    $draftModuleLinks = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $finalSubmodulesData = [];

    // Process each link to get actual submodule and study material details
    foreach ($draftModuleLinks as $link) {
        $module_id = $link['module_id'];
        $module_name = $link['module_name'];
        $course_draft_id = $link['course_draft_id'];
        $draft_module_id = $link['draft_module_id'];
        $submodule_ids_csv = $link['submodule_ids_csv'];

        // If there are no submodules linked, skip this entry or handle as needed
        if (empty($submodule_ids_csv)) {
            continue;
        }

        $submoduleIds = array_map('intval', array_filter(explode(',', $submodule_ids_csv)));

        // Fetch details for each submodule ID
        if (!empty($submoduleIds)) {
            $placeholders = implode(',', array_fill(0, count($submoduleIds), '?'));
            $stmtSubmodule = $pdo->prepare("
                SELECT submodule_id, submodule_name, study_material
                FROM submodule_master
                WHERE submodule_id IN ($placeholders)
            ");
            $stmtSubmodule->execute($submoduleIds);
            $submoduleDetails = $stmtSubmodule->fetchAll(PDO::FETCH_ASSOC);

            foreach ($submoduleDetails as $submodule) {
                $contents = [];

                if (!empty($submodule['study_material'])) {
                    $materialIds = array_map('intval', array_filter(explode(',', $submodule['study_material'])));

                    if (!empty($materialIds)) {
                        $materialPlaceholders = implode(',', array_fill(0, count($materialIds), '?'));
                        $stmtMaterial = $pdo->prepare("
                            SELECT study_material_id, title, actual_file_name, vimeo_link
                            FROM study_material
                            WHERE study_material_id IN ($materialPlaceholders)
                        ");
                        $stmtMaterial->execute($materialIds);
                        $contents = $stmtMaterial->fetchAll(PDO::FETCH_ASSOC);
                    }
                }

                // Add the full details for each submodule, including the module info
                $finalSubmodulesData[] = [
                    'draft_module_id' => $draft_module_id,
                    'course_draft_id' => $course_draft_id,
                    'module_id'       => $module_id,
                    'module_name'     => $module_name,
                    // Note: We're only returning one submodule_id at a time now as it's looped
                    'submodule_id'    => $submodule['submodule_id'],
                    'submodule_name'  => $submodule['submodule_name'],
                    'study_material'  => $submodule['study_material'], // Keep the raw string if needed
                    'contents'        => $contents
                ];
            }
        }
    }

    echo json_encode(['success' => true, 'submodules' => $finalSubmodulesData]);

} catch (Exception $e) {
    error_log("Error in get_submodules_by_draft.php: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Server error: ' . $e->getMessage()]);
}