<?php
require_once('../includes/connection.php');
header('Content-Type: application/json');

if (!isset($_GET['assignment_id'])) {
    echo json_encode(["status" => "error", "message" => "Assignment ID is required"]);
    exit();
}

$assignmentId = intval($_GET['assignment_id']);
$courseId = intval($_GET['course_id']);
$user_id = $_SESSION['user_id'];
$attemptNo = isset($_GET['attempt']) ? intval($_GET['attempt']) : null;

try {
    // Fetch assignment details
    $sql = "SELECT * FROM assignment_master WHERE assignment_id = :assignment_id";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':assignment_id', $assignmentId, PDO::PARAM_INT);
    $stmt->execute();
    $assignment = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$assignment) {
        echo json_encode(["status" => "error", "message" => "Assignment not found."]);
        exit;
    }

    // 🔹 Fetch attempt (specific or latest)
    if ($attemptNo) {
        $stmt = $pdo->prepare("
            SELECT *
            FROM assignment_attempts
            WHERE assignment_id = ? AND user_id = ? AND attempt_count = ? AND course_id = ?
            LIMIT 1
        ");
        $stmt->execute([$assignmentId, $user_id, $attemptNo, $courseId]);
    } else {
        $stmt = $pdo->prepare("
            SELECT *
            FROM assignment_attempts
            WHERE assignment_id = ? AND user_id = ?
            ORDER BY assignment_attempt_id DESC
            LIMIT 1
        ");
        $stmt->execute([$assignmentId, $user_id]);
    }

    $attempt = $stmt->fetch(PDO::FETCH_ASSOC);

    // Default values
    $totalMarks = $assignment['total_marks'];
    $obtainedMarks = 0; // ✅ default
    $attemptCount = $attemptNo ?? 0;

    if ($attempt) {
        $totalMarks = $attempt['total_marks'];
        $obtainedMarks = $attempt['obtained_marks']; // ✅ fetched from table
        $attemptCount = $attempt['attempt_count'];
    }

    $questionIds = array_filter(array_map('trim', explode(',', $assignment['question_ids'])));
    if (empty($questionIds)) {
        echo json_encode([
            "status"           => $attempt ? "attempted" : "success",
            "assignment_title" => $assignment['assignment_title'],
            "total_marks"      => $totalMarks,
            "obtained_marks"   => $obtainedMarks,
            "attempt_count"    => $attemptCount,
            "questions"        => []
        ]);
        exit;
    }

    // Fetch questions
    $placeholders = rtrim(str_repeat('?,', count($questionIds)), ',');
    $qSql = "SELECT * FROM question_master WHERE question_master_id IN ($placeholders)";
    $qStmt = $pdo->prepare($qSql);
    $qStmt->execute($questionIds);
    $questions = $qStmt->fetchAll(PDO::FETCH_ASSOC);

    // Fetch selected answers for this attempt
    $qaSql = "SELECT question_id, selected_answer FROM assignment_attempts_data WHERE assignment_attempt_id = ?";
    $qaStmt = $pdo->prepare($qaSql);
    if ($attempt) {
        $qaStmt->execute([$attempt['assignment_attempt_id']]);
        $attemptData = $qaStmt->fetchAll(PDO::FETCH_ASSOC);
    } else {
        $attemptData = [];
    }

    // Map answers
    $selectedAnswersMap = [];
    foreach ($attemptData as $row) {
        $selectedAnswersMap[$row['question_id']] = $row['selected_answer'];
    }

    // Format response
    $formattedQuestions = array_map(function ($q) use ($selectedAnswersMap) {
        $options = [
            'option_1' => $q['option_1'],
            'option_2' => $q['option_2'],
            'option_3' => $q['option_3'],
            'option_4' => $q['option_4'],
        ];

        $type = strtolower(trim($q['type']));
        $correctAnswerValues = [];

        if ($type === 'yes_no' || $type === 'true_false') {
            $val = strtolower(trim($q['correct_answer']));
            if (!empty($val)) {
                $correctAnswerValues[] = ucfirst($val);
            }
        } else {
            $correctAnswerKeys = array_map('trim', explode(',', $q['correct_answer']));
            foreach ($correctAnswerKeys as $key) {
                $keyFormatted = strtolower(str_replace(' ', '_', $key));
                if (isset($options[$keyFormatted])) {
                    $correctAnswerValues[] = $options[$keyFormatted];
                }
            }
        }

        // Selected answers
        $selectedAnswerRaw = $selectedAnswersMap[$q['question_master_id']] ?? null;
        $selectedAnswerValues = [];

        if ($selectedAnswerRaw) {
            if ($type === 'yes_no' || $type === 'true_false') {
                $selectedAnswerValues[] = ucfirst(strtolower($selectedAnswerRaw));
            } else {
                $keys = array_map('trim', explode(',', $selectedAnswerRaw));
                foreach ($keys as $key) {
                    $keyFormatted = strtolower(str_replace(' ', '_', $key));
                    if (isset($options[$keyFormatted])) {
                        $selectedAnswerValues[] = $options[$keyFormatted];
                    }
                }
            }
        }

        return [
            "question_id"      => $q["question_master_id"],
            "type"             => $q["type"],
            "question"         => $q["question"],
            "media"            => [
                "image"      => $q["image_url"],
                "video"      => $q["video_url"],
                "video_type" => $q["video_type"]
            ],
            "options"          => array_values(array_filter($options)),
            "correct_answer"   => $correctAnswerValues,
            "selected_answer"  => $selectedAnswerValues,
            "hint"             => $q["hint"],
            "marks"            => $q["marks"]
        ];
    }, $questions);

    // Final JSON
    echo json_encode([
        "status"           => $attempt ? "attempted" : "success",
        "assignment_title" => $assignment['assignment_title'],
        "total_marks"      => $totalMarks,
        "obtained_marks"   => $obtainedMarks,  // ✅ added here
        "attempt_count"    => $attemptCount,
        "questions"        => $formattedQuestions
    ]);

} catch (PDOException $e) {
    echo json_encode(["status" => "error", "message" => "Database Error: " . $e->getMessage()]);
}
