<?php
require_once('../includes/connection.php');

header('Content-Type: application/json');
date_default_timezone_set('Asia/Kolkata');

$user_id = $_SESSION['user_id'] ?? 0;

if (!$user_id) {
    echo json_encode(['status' => 'error', 'message' => 'Missing user']);
    exit;
}

try {
    $stmt = $pdo->prepare("
        SELECT ucp.last_content_index, 
               ucp.max_completed_index, 
               ucp.course_id, 
               cm.course_title,
               cm.start_date,
               cm.end_date
        FROM user_course_progress ucp
        LEFT JOIN course_master cm 
               ON ucp.course_id = cm.course_id
        WHERE ucp.user_id = :user_id
        ORDER BY ucp.id DESC 
        LIMIT 1
    ");
    $stmt->execute([':user_id' => $user_id]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($result) {
        $course_id = $result['course_id'];
        $start_date = $result['start_date'];
        $end_date   = $result['end_date'];

        // ✅ Days left until end_date
        $days_left = null;
        if (!empty($end_date)) {
            $today = new DateTime('today');
            $end   = new DateTime($end_date);
            $days_left = ($end >= $today) ? $today->diff($end)->days : 0;
        }

        // -------------------------------------------------
        // Step 1: Get modules assigned to course
        // -------------------------------------------------
        $stmtModules = $pdo->prepare("
            SELECT mm.study_material
            FROM course_module cm
            JOIN module_master mm ON cm.module_id = mm.module_id
            WHERE cm.course_id = :course_id AND mm.study_material != ''
        ");
        $stmtModules->execute([':course_id' => $course_id]);
        $materialsFromModules = $stmtModules->fetchAll(PDO::FETCH_COLUMN);

        // Step 2: Get submodules assigned to course
        $stmtSubModules = $pdo->prepare("
            SELECT sm.study_material
            FROM course_submodule csm
            JOIN submodule_master sm ON csm.submodule_id = sm.submodule_id
            WHERE csm.course_id = :course_id AND sm.study_material != ''
        ");
        $stmtSubModules->execute([':course_id' => $course_id]);
        $materialsFromSubModules = $stmtSubModules->fetchAll(PDO::FETCH_COLUMN);

        // Step 3: Merge all study_material IDs
        $allMaterialIds = [];
        foreach (array_merge($materialsFromModules, $materialsFromSubModules) as $materialStr) {
            $ids = array_filter(array_map('trim', explode(',', $materialStr)));
            $allMaterialIds = array_merge($allMaterialIds, $ids);
        }

        // Step 4: Keep unique IDs only
        $allMaterialIds = array_unique($allMaterialIds);

        // Step 5: Count only study materials that exist in stud_nmaterial
        $studyMaterialCount = 0;
        if (!empty($allMaterialIds)) {
            $placeholders = implode(',', array_fill(0, count($allMaterialIds), '?'));
            $stmt = $pdo->prepare("
                SELECT COUNT(*) 
                FROM study_material 
                WHERE study_material_id IN ($placeholders)
            ");
            $stmt->execute($allMaterialIds);
            $studyMaterialCount = (int)$stmt->fetchColumn();
        }

        // Quizzes
        $stmtQuiz = $pdo->prepare("SELECT COUNT(*) FROM course_quiz WHERE course_id = :course_id");
        $stmtQuiz->execute([':course_id' => $course_id]);
        $quizCount = $stmtQuiz->fetchColumn();

        // Assignments
        $stmtAssign = $pdo->prepare("SELECT COUNT(*) FROM course_assignment WHERE course_id = :course_id");
        $stmtAssign->execute([':course_id' => $course_id]);
        $assignmentCount = $stmtAssign->fetchColumn();

        // Total course items
        $totalItems = $studyMaterialCount + $quizCount + $assignmentCount;

        // ✅ Completed items (fix: +1 for zero-based index)
        $maxCompletedIndex = isset($result['max_completed_index']) ? (int)$result['max_completed_index'] : -1;
        $completedItems = ($maxCompletedIndex >= 0) ? $maxCompletedIndex + 1 : 0;

        // ✅ Completed percentage
        $completed_percent = ($totalItems > 0)
            ? round(($completedItems / $totalItems) * 100)
            : 0;

        $response = [
            'course_id'         => $course_id,
            'course_title'      => $result['course_title'],
            'days_left'         => $days_left,
            'completed_percent' => $completed_percent,
            'start_date'        => $start_date,
            'end_date'          => $end_date,
            'progress_breakdown' => [
                'study_materials' => $studyMaterialCount,
                'quizzes'         => $quizCount,
                'assignments'     => $assignmentCount,
                'total'           => $totalItems,
                'completed'       => $completedItems
            ]
        ];
    } else {
        $response = 'NOTSTARTED';
    }

    echo json_encode(['status' => 'success', 'data' => $response]);

} catch (PDOException $e) {
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}
