<?php
require_once('../includes/connection.php');
header('Content-Type: application/json');

try {
    $input = json_decode(file_get_contents("php://input"), true);
    if (!$input) {
        throw new Exception("Invalid JSON input");
    }

    $assignment_id = intval($input['assignment_id'] ?? 0);
    $user_id       = $_SESSION['user_id'];
    $course_id     = intval($input['course_id'] ?? 0);
    $institute_id  = $_SESSION['institute_id'];
    $questions     = $input['questions'] ?? [];

    if (!$assignment_id || !$course_id) {
        throw new Exception("Missing required fields");
    }

    // Check existing attempts
    $stmtCheck = $pdo->prepare("
        SELECT COUNT(*) as attempt_count 
        FROM assignment_attempts 
        WHERE assignment_id = :assignment_id AND user_id = :user_id AND course_id = :course_id
    ");
    $stmtCheck->execute([
        ':assignment_id' => $assignment_id,
        ':user_id'       => $user_id,
        ':course_id'       => $course_id
    ]);
    $attemptCount = $stmtCheck->fetchColumn();

    if ($attemptCount >= 3) {
        throw new Exception("Maximum 3 attempts allowed");
    }

    // === Calculate total marks and obtained marks dynamically ===
    $total_marks = 0;
    $obtained_marks = 0;

    foreach ($questions as $q) {
        // Each question has its own marks
        $qMarks = isset($q['marks']) ? floatval($q['marks']) : 1; // default 1 mark per question
        $total_marks += $qMarks;

        $correct_answer  = trim($q['correct_answer'] ?? "");
        $selected_answer = trim($q['selected_answer'] ?? "");

        // compare multi-answer safely
        $corrArr = array_map('trim', array_map('strtolower', explode(',', $correct_answer)));
        $selArr  = array_map('trim', array_map('strtolower', explode(',', $selected_answer)));
        sort($corrArr);
        sort($selArr);

        if ($corrArr === $selArr) {
            $obtained_marks += $qMarks;
        }
    }

    // Insert into assignment_attempts
    $stmt = $pdo->prepare("
        INSERT INTO assignment_attempts 
        (assignment_id, user_id, course_id, institute_id, total_marks, obtained_marks, attempt_count) 
        VALUES 
        (:assignment_id, :user_id, :course_id, :institute_id, :total_marks, :obtained_marks, :attempt_number)
    ");
    $stmt->execute([
        ':assignment_id'  => $assignment_id,
        ':user_id'        => $user_id,
        ':course_id'      => $course_id,
        ':institute_id'   => $institute_id,
        ':total_marks'    => $total_marks,
        ':obtained_marks' => $obtained_marks,
        ':attempt_number' => $attemptCount + 1
    ]);

    $attempt_id = $pdo->lastInsertId();

    // Insert details per question
    $stmtDetail = $pdo->prepare("
        INSERT INTO assignment_attempts_data 
        (assignment_attempt_id, question_id, correct_answer, selected_answer, assignment_id, user_id, course_id) 
        VALUES 
        (:attempt_id, :question_id, :correct_answer, :selected_answer, :assignment_id, :user_id, :course_id)
    ");

    foreach ($questions as $q) {
        $stmtDetail->execute([
            ':attempt_id'     => $attempt_id,
            ':question_id'    => intval($q['question_id']),
            ':correct_answer' => trim($q['correct_answer'] ?? ""),
            ':selected_answer'=> trim($q['selected_answer'] ?? ""),
            ':assignment_id'  => $assignment_id,
            ':user_id'        => $user_id,
            ':course_id'      => $course_id
        ]);
    }
    

        $percentage = $total_marks > 0 ? ($obtained_marks / $total_marks) * 100 : 0;
        
        // ✅ Update user_course_progress if 3rd attempt OR percentage >= 70%
        $finalAttemptNum = $attemptCount + 1;
        if ($finalAttemptNum == 3 || $percentage >= 70) {
            $stmtUpdate = $pdo->prepare("
                UPDATE user_course_progress 
                SET max_completed_index = last_content_index 
                WHERE user_id = :user_id AND institute_id = :institute_id AND course_id = :course_id
            ");
            $stmtUpdate->execute([
                ':user_id'      => $user_id,
                ':institute_id' => $institute_id,
                ':course_id'    => $course_id
            ]);
        }


    

    echo json_encode([
        "status" => "success",
        "message" => "Assignment attempt saved successfully",
        "attempt_id" => $attempt_id,
        "attempt_number" => $attemptCount + 1,
        "obtained_marks" => $obtained_marks,
        "total_marks" => $total_marks
    ]);

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        "status" => "error",
        "message" => $e->getMessage()
    ]);
}
