<?php
require_once ('../includes/connection.php');

header('Content-Type: application/json');

// Receive the input data from the POST request
$data = json_decode(file_get_contents("php://input"), true);

// Check if category_name is set and is an array
if (isset($data['category_name']) && is_array($data['category_name']) && !empty($data['category_name'])) {
    $categoryNames = array_map('trim', $data['category_name']); // Remove extra whitespace from each category name
} else {
    echo json_encode(["status" => "error", "message" => "At least one category name is required"]);
    exit();
}

try {
    $user_id = $_SESSION['user_id'];
    $institute_id = $_SESSION['institute_id'];

    // Prepare the SQL query to check if the category already exists
    $checkSql = "SELECT COUNT(*) FROM category_master WHERE category_name = :category_name AND institute_id = :institute_id";
    $checkStmt = $pdo->prepare($checkSql);

    $existingCategories = []; // To store the categories that already exist
    $categoriesToInsert = [];  // To store categories that do not exist yet and need to be inserted

    foreach ($categoryNames as $categoryName) {
        $checkStmt->bindParam(':category_name', $categoryName);
        $checkStmt->bindParam(':institute_id', $institute_id);
        $checkStmt->execute();
        $count = $checkStmt->fetchColumn();

        if ($count > 0) {
            // If the category already exists, add it to the existingCategories array
            $existingCategories[] = $categoryName;
        } else {
            // If the category doesn't exist, add it to the categoriesToInsert array
            $categoriesToInsert[] = $categoryName;
        }
    }

    // If there are categories to insert, proceed with the insertion
    if (!empty($categoriesToInsert)) {
        $insertSql = "INSERT INTO category_master (category_name, institute_id, created_by) VALUES (:category_name, :institute_id, :user_id)";
        $insertStmt = $pdo->prepare($insertSql);

        foreach ($categoriesToInsert as $categoryName) {
            $insertStmt->bindParam(':category_name', $categoryName);
            $insertStmt->bindParam(':user_id', $user_id);
            $insertStmt->bindParam(':institute_id', $institute_id);
            $insertStmt->execute();
        }
    }

    if (empty($categoriesToInsert) && !empty($existingCategories)) {
        // If no categories were inserted (all existed already)
        echo json_encode([
            "status" => "error",
            "message" => "The following categories already exist: " . implode(", ", $existingCategories)
        ]);
    } elseif (!empty($categoriesToInsert)) {
        // If new categories were inserted, but some were duplicates
        $responseMessage = "Category created successfully";
        
        if (!empty($existingCategories)) {
            $responseMessage .= ". The following categories already exist: " . implode(", ", $existingCategories);
        }

        echo json_encode([
            "status" => "success",
            "message" => $responseMessage
        ]);
    } else {
        echo json_encode([
            "status" => "error",
            "message" => "Something went wrong"
        ]);
    }

} catch (PDOException $e) {
    echo json_encode(["status" => "error", "message" => "Error: " . $e->getMessage()]);
}

?>