<?php
header('Content-Type: application/json');
require_once('../includes/connection.php'); // Adjust path to your PDO database connection file

$response = ['success' => false, 'message' => ''];

try {
    if (!isset($pdo) || !$pdo instanceof PDO) {
        throw new Exception("PDO connection object not found or invalid.");
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $draftId = filter_input(INPUT_POST, 'draft_id', FILTER_VALIDATE_INT);
        $moduleIds = $_POST['module_ids'] ?? []; // Expected to be an array

        // Validate inputs
        if ($draftId === false || $draftId === null) {
            $response['message'] = 'Invalid Draft ID.';
        } elseif (!is_array($moduleIds) || empty($moduleIds)) {
            $response['message'] = 'No modules selected.';
        } else {
            $addedCount = 0;
            $failedCount = 0;
            $messages = [];

            // Begin transaction for atomicity
            $pdo->beginTransaction();

            // Use INSERT IGNORE to prevent adding duplicates if a unique constraint exists
            // on (course_draft_id, module_id)
            $insertSql = "INSERT IGNORE INTO course_draft_modules (course_draft_id, module_id) VALUES (:course_draft_id, :module_id)";
            $stmt = $pdo->prepare($insertSql);

            foreach ($moduleIds as $moduleId) {
                $moduleId = filter_var($moduleId, FILTER_VALIDATE_INT);
                if ($moduleId !== false && $moduleId !== null) {
                    $stmt->bindParam(':course_draft_id', $draftId, PDO::PARAM_INT);
                    $stmt->bindParam(':module_id', $moduleId, PDO::PARAM_INT);
                    if ($stmt->execute()) {
                        if ($stmt->rowCount() > 0) {
                            $addedCount++;
                        } else {
                            // rowCount 0 means it was ignored (duplicate)
                            $messages[] = "Module ID {$moduleId} was already linked.";
                        }
                    } else {
                        $failedCount++;
                        $messages[] = "Failed to add Module ID {$moduleId}: " . implode(", ", $stmt->errorInfo());
                    }
                } else {
                    $failedCount++;
                    $messages[] = "Invalid Module ID provided.";
                }
            }

            $pdo->commit(); // Commit the transaction

            if ($addedCount > 0) {
                $response['success'] = true;
                $response['message'] = "Successfully added {$addedCount} module(s). " . implode(" ", $messages);
            } elseif ($failedCount > 0) {
                $response['message'] = "Errors occurred while adding modules. " . implode(" ", $messages);
            } else {
                $response['success'] = true; // All were duplicates, but not an error
                $response['message'] = "All selected modules were already linked. " . implode(" ", $messages);
            }

        }
    } else {
        $response['message'] = 'Invalid request method. Only POST requests are allowed.';
    }

} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack(); // Rollback on database error
    }
    $response['message'] = 'Database error: ' . $e->getMessage();
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack(); // Rollback on general application error
    }
    $response['message'] = 'Application error: ' . $e->getMessage();
}

echo json_encode($response);
?>