<?php
// Start the session
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

require_once('../includes/connection.php'); // Adjust path as needed

// Set header for JSON response
header('Content-Type: application/json');

$response = ['success' => false, 'message' => '', 'action' => ''];

try {
    // Validate database connection
    if (!isset($pdo) || !$pdo instanceof PDO) {
        throw new Exception("Database connection not available");
    }

    // Check for required parameters
    if (!isset($_POST['draft_id']) || empty($_POST['draft_id'])) {
        throw new Exception("Draft ID is required");
    }

    if (!isset($_POST['submodules']) || !is_array($_POST['submodules']) || count($_POST['submodules']) === 0) {
        throw new Exception("No submodules selected");
    }

    $draft_id = (int)$_POST['draft_id'];
    $submodules = $_POST['submodules'];

    // Begin transaction
    $pdo->beginTransaction();

    // We'll process one module at a time (assuming all submodules belong to same module in one request)
    $module_id = null;
    $submodule_ids = [];

    foreach ($submodules as $submoduleData) {
        if (!isset($submoduleData['submodule_id']) || !isset($submoduleData['module_id'])) {
            continue; // Skip invalid entries
        }

        if ($module_id === null) {
            $module_id = (int)$submoduleData['module_id'];
        } elseif ($module_id !== (int)$submoduleData['module_id']) {
            throw new Exception("All submodules must belong to the same module in a single request");
        }

        $submodule_ids[] = (int)$submoduleData['submodule_id'];
    }

    if (empty($submodule_ids)) {
        throw new Exception("No valid submodule IDs provided");
    }

    // Check if this module already exists in the draft
    $checkSql = "SELECT draft_module_id, sub_modules FROM course_draft_modules 
                 WHERE course_draft_id = :draft_id AND module_id = :module_id";
    $checkStmt = $pdo->prepare($checkSql);
    $checkStmt->execute([':draft_id' => $draft_id, ':module_id' => $module_id]);
    $existingRecord = $checkStmt->fetch(PDO::FETCH_ASSOC);

    if ($existingRecord) {
        // Update existing record - merge submodules
        $existing_submodules = $existingRecord['sub_modules'] ? explode(',', $existingRecord['sub_modules']) : [];
        $merged_submodules = array_unique(array_merge($existing_submodules, $submodule_ids));
        $merged_submodules_str = implode(',', $merged_submodules);

        $updateSql = "UPDATE course_draft_modules 
                      SET sub_modules = :sub_modules 
                      WHERE draft_module_id = :draft_module_id";
        $updateStmt = $pdo->prepare($updateSql);
        $updateStmt->execute([
            ':sub_modules' => $merged_submodules_str,
            ':draft_module_id' => $existingRecord['draft_module_id']
        ]);

        $response['action'] = 'updated';
    } else {
        // Insert new record
        $sub_modules_str = implode(',', $submodule_ids);

        $insertSql = "INSERT INTO course_draft_modules 
                     (course_draft_id, module_id, sub_modules) 
                     VALUES (:draft_id, :module_id, :sub_modules)";
        $insertStmt = $pdo->prepare($insertSql);
        $insertStmt->execute([
            ':draft_id' => $draft_id,
            ':module_id' => $module_id,
            ':sub_modules' => $sub_modules_str
        ]);

        $response['action'] = 'inserted';
    }

    $pdo->commit();
    $response['success'] = true;
    $response['message'] = "Submodules successfully processed";

} catch (PDOException $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $response['message'] = "Database error: " . $e->getMessage();
    error_log("PDOException in add_course_draft_submodules.php: " . $e->getMessage());
} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $response['message'] = "Error: " . $e->getMessage();
    error_log("Exception in add_course_draft_submodules.php: " . $e->getMessage());
}

echo json_encode($response);
?>