<?php
header('Content-Type: application/json');
require_once('../includes/connection.php');

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Get POST data
$course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
$status = isset($_POST['status']) ? trim($_POST['status']) : '';
$comment = isset($_POST['comment']) ? trim($_POST['comment']) : '';

// Validate inputs
if (!$course_id) {
    echo json_encode(['success' => false, 'message' => 'Course ID is required.']);
    exit;
}

if (empty($status)) {
    echo json_encode(['success' => false, 'message' => 'Please select approval status.']);
    exit;
}

if (!in_array($status, ['approved', 'rejected'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid status selected.']);
    exit;
}

// Validate comment for rejection
if ($status == 'rejected' && empty($comment)) {
    echo json_encode(['success' => false, 'message' => 'Comment is required for rejection.']);
    exit;
}

// Get current user ID
$expert_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;

if (!$expert_id) {
    echo json_encode(['success' => false, 'message' => 'User session not found.']);
    exit;
}

try {
    // Check if course exists
    $checkStmt = $pdo->prepare("SELECT course_id FROM course_master WHERE course_id = :id AND is_block = 0");
    $checkStmt->bindParam(':id', $course_id, PDO::PARAM_INT);
    $checkStmt->execute();
    $course = $checkStmt->fetch(PDO::FETCH_ASSOC);

    if (!$course) {
        echo json_encode(['success' => false, 'message' => 'Course not found or is blocked.']);
        exit;
    }

    // Update course with approval/rejection status
    // Using common field names - adjust if your database uses different field names
    $updateSql = "UPDATE course_master SET 
                    expert_approval_status = :status,
                    expert_approval_comment = :comment,
                    expert_approved_by = :expert_id,
                    expert_approved_at = NOW()
                  WHERE course_id = :course_id";

    $updateStmt = $pdo->prepare($updateSql);
    $updateStmt->bindParam(':status', $status, PDO::PARAM_STR);
    $updateStmt->bindParam(':comment', $comment, PDO::PARAM_STR);
    $updateStmt->bindParam(':expert_id', $expert_id, PDO::PARAM_INT);
    $updateStmt->bindParam(':course_id', $course_id, PDO::PARAM_INT);
    
    $updateStmt->execute();

    if ($updateStmt->rowCount() > 0) {
        $message = $status == 'approved' ? 'Course approved successfully.' : 'Course rejected successfully.';
        echo json_encode(['success' => true, 'message' => $message]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to update course approval status.']);
    }
} catch (PDOException $e) {
    // Check if error is due to missing columns
    if (strpos($e->getMessage(), "Unknown column") !== false) {
        // Try alternative field names
        try {
            $updateSql = "UPDATE course_master SET 
                            approval_status = :status,
                            approval_comment = :comment,
                            approved_by = :expert_id,
                            approved_at = NOW()
                          WHERE course_id = :course_id";

            $updateStmt = $pdo->prepare($updateSql);
            $updateStmt->bindParam(':status', $status, PDO::PARAM_STR);
            $updateStmt->bindParam(':comment', $comment, PDO::PARAM_STR);
            $updateStmt->bindParam(':expert_id', $expert_id, PDO::PARAM_INT);
            $updateStmt->bindParam(':course_id', $course_id, PDO::PARAM_INT);
            
            $updateStmt->execute();

            if ($updateStmt->rowCount() > 0) {
                $message = $status == 'approved' ? 'Course approved successfully.' : 'Course rejected successfully.';
                echo json_encode(['success' => true, 'message' => $message]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to update course approval status.']);
            }
        } catch (PDOException $e2) {
            echo json_encode(['success' => false, 'message' => 'Database error: Please contact administrator. Column names may need to be configured.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    }
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'An error occurred: ' . $e->getMessage()]);
}
?>

