<?php
require_once('../includes/connection.php');
require_once('emailer.php');
header('Content-Type: application/json');

// Manually load PSR SimpleCache interfaces first (before PhpSpreadsheet)
$psrBaseDir = __DIR__ . '/../../PHPOffice/src/Psr/SimpleCache/';
$psrFiles = [
    'CacheException.php',
    'InvalidArgumentException.php',
    'CacheInterface.php'
];

foreach ($psrFiles as $file) {
    $filePath = $psrBaseDir . $file;
    if (file_exists($filePath)) {
        require_once($filePath);
    } else {
        error_log("PSR SimpleCache file not found: $filePath");
    }
}

// Verify CacheInterface is loaded - critical for PhpSpreadsheet
if (!interface_exists('Psr\SimpleCache\CacheInterface')) {
    echo json_encode([
        "status" => "error", 
        "message" => "System error: Required library files not found. Please contact administrator."
    ]);
    exit();
}

// Setup autoloader for PhpSpreadsheet and PSR SimpleCache
spl_autoload_register(function ($class) {
    // Handle PSR SimpleCache
    if (strpos($class, 'Psr\\SimpleCache\\') === 0) {
        $base_dir = __DIR__ . '/../../PHPOffice/src/';
        $relative_class = substr($class, strlen('Psr\\'));
        $file = $base_dir . str_replace('\\', '/', $relative_class) . '.php';
        if (file_exists($file)) {
            require $file;
            return;
        }
    }
    
    // Handle PhpSpreadsheet
    $prefix = 'PhpOffice\\PhpSpreadsheet\\';
    $base_dir = __DIR__ . '/../../PHPOffice/src/PhpSpreadsheet/';
    
    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }
    
    $relative_class = substr($class, $len);
    $file = $base_dir . str_replace('\\', '/', $relative_class) . '.php';
    
    if (file_exists($file)) {
        require $file;
    }
});

// Load PhpSpreadsheet Settings and set cache before using IOFactory
require_once(__DIR__ . '/../../PHPOffice/src/PhpSpreadsheet/Settings.php');
require_once(__DIR__ . '/../../PHPOffice/src/PhpSpreadsheet/Collection/Memory/SimpleCache1.php');

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Collection\Memory\SimpleCache1;
use PhpOffice\PhpSpreadsheet\Settings;

// Set cache before using PhpSpreadsheet - MUST be done before any PhpSpreadsheet operations
try {
    $cache = new SimpleCache1();
    Settings::setCache($cache);
} catch (Exception $e) {
    error_log("Error setting PhpSpreadsheet cache: " . $e->getMessage());
    // Continue anyway - might work without explicit cache
}

/**
 * Normalize date to YYYY-MM-DD format
 * Handles various date formats from Excel/CSV
 */
function normalizeDate($dateInput) {
    if (empty($dateInput)) {
        return '';
    }
    
    // Remove any whitespace
    $dateInput = trim($dateInput);
    
    // If it's already in YYYY-MM-DD format, validate and return
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateInput)) {
        $parts = explode('-', $dateInput);
        if (checkdate($parts[1], $parts[2], $parts[0])) {
            return $dateInput;
        }
    }
    
    // Try to parse as Excel date serial number (if it's a number)
    if (is_numeric($dateInput)) {
        // Excel date serial number (days since 1900-01-01)
        // Convert to Unix timestamp
        $excelEpoch = mktime(0, 0, 0, 1, 1, 1900);
        $unixTimestamp = $excelEpoch + (($dateInput - 1) * 86400);
        $date = date('Y-m-d', $unixTimestamp);
        if ($date && $date !== '1970-01-01') {
            return $date;
        }
    }
    
    // Try various date formats
    $formats = [
        'Y-m-d',           // 2000-01-15
        'Y/m/d',           // 2000/01/15
        'd-m-Y',           // 15-01-2000
        'd/m/Y',           // 15/01/2000
        'm-d-Y',           // 01-15-2000
        'm/d/Y',           // 01/15/2000
        'd.m.Y',           // 15.01.2000
        'Y.m.d',           // 2000.01.15
    ];
    
    foreach ($formats as $format) {
        $date = DateTime::createFromFormat($format, $dateInput);
        if ($date && $date->format($format) === $dateInput) {
            return $date->format('Y-m-d');
        }
    }
    
    // Try strtotime as last resort
    $timestamp = strtotime($dateInput);
    if ($timestamp !== false) {
        $date = date('Y-m-d', $timestamp);
        // Validate the date is reasonable (not too far in future/past)
        $year = (int)date('Y', $timestamp);
        if ($year >= 1900 && $year <= 2100) {
            return $date;
        }
    }
    
    return ''; // Invalid date
}

/**
 * Generate error report file with failed records
 */
function generateErrorReport($errorRecords, $headerRow) {
    $errorDir = __DIR__ . '/../error_reports/';
    if (!file_exists($errorDir)) {
        mkdir($errorDir, 0755, true);
    }
    
    $filename = 'error_report_' . date('Y-m-d_His') . '_' . uniqid() . '.csv';
    $filepath = $errorDir . $filename;
    
    $output = fopen($filepath, 'w');
    
    // Write UTF-8 BOM for Excel compatibility
    fprintf($output, "\xEF\xBB\xBF");
    
    // Write headers with Error Message column
    $errorHeaders = array_merge($headerRow, ['Error Message']);
    fputcsv($output, $errorHeaders, ',', '"');
    
    // Write error records
    foreach ($errorRecords as $record) {
        $row = [
            $record['student_name'],
            $record['email'],
            $record['phone'],
            $record['gender'],
            $record['dob'],
            $record['institute_name'],
            $record['error_message']
        ];
        fputcsv($output, $row, ',', '"');
    }
    
    fclose($output);
    
    // Return relative path for download
    return 'error_reports/' . $filename;
}

// Check if file is uploaded
if (!isset($_FILES['upload_file']) || $_FILES['upload_file']['error'] !== UPLOAD_ERR_OK) {
    echo json_encode(["status" => "error", "message" => "Please select a valid file to upload."]);
    exit();
}

$uploadedFile = $_FILES['upload_file'];
$fileExtension = strtolower(pathinfo($uploadedFile['name'], PATHINFO_EXTENSION));

// Validate file extension
$allowedExtensions = ['csv', 'xlsx', 'xls'];
if (!in_array($fileExtension, $allowedExtensions)) {
    echo json_encode(["status" => "error", "message" => "Invalid file format. Please upload CSV or Excel file."]);
    exit();
}

// Validate file size (5MB max)
$maxSize = 5 * 1024 * 1024; // 5MB
if ($uploadedFile['size'] > $maxSize) {
    echo json_encode(["status" => "error", "message" => "File size exceeds 5MB limit."]);
    exit();
}

try {
    // Ensure cache is set before loading file - this is critical
    try {
        if (!Settings::getCache()) {
            $cache = new SimpleCache1();
            Settings::setCache($cache);
        }
    } catch (Exception $cacheError) {
        error_log("Cache setup error: " . $cacheError->getMessage());
        // Try to set cache anyway
        try {
            $cache = new SimpleCache1();
            Settings::setCache($cache);
        } catch (Exception $e) {
            error_log("Failed to set cache: " . $e->getMessage());
        }
    }
    
    // Load the uploaded file
    $inputFileName = $uploadedFile['tmp_name'];
    $spreadsheet = IOFactory::load($inputFileName);
    $worksheet = $spreadsheet->getActiveSheet();
    
    // Get all rows as array - get formatted values for dates
    $rows = [];
    $highestRow = $worksheet->getHighestRow();
    $highestColumn = $worksheet->getHighestColumn();
    
    for ($row = 1; $row <= $highestRow; $row++) {
        $rowData = [];
        for ($col = 'A'; $col <= $highestColumn; $col++) {
            $cell = $worksheet->getCell($col . $row);
            $value = $cell->getFormattedValue(); // Get formatted value (handles dates properly)
            
            // If value is still numeric and looks like Excel date serial, convert it
            if (is_numeric($value) && $value > 0 && $value < 100000 && strlen((string)$value) <= 6) {
                // Likely an Excel date serial number (dates are typically 20000-50000 range)
                try {
                    // Excel epoch is 1900-01-01, but Excel incorrectly treats 1900 as leap year
                    $excelEpoch = mktime(0, 0, 0, 1, 1, 1900);
                    $unixTimestamp = $excelEpoch + (($value - 2) * 86400); // -2 because Excel's epoch bug
                    $convertedDate = date('Y-m-d', $unixTimestamp);
                    // Validate it's a reasonable date
                    $year = (int)date('Y', $unixTimestamp);
                    if ($year >= 1900 && $year <= 2100) {
                        $value = $convertedDate;
                    }
                } catch (Exception $e) {
                    // Keep original value if conversion fails
                }
            }
            
            $rowData[] = $value;
        }
        $rows[] = $rowData;
    }
    
    // Check if file has data
    if (empty($rows) || count($rows) < 2) {
        echo json_encode(["status" => "error", "message" => "File is empty or has no data rows."]);
        exit();
    }
    
    // Remove header row (first row)
    $headerRow = array_shift($rows);
    
    // Validate header columns
    $expectedHeaders = ['Student Name', 'Email Address', 'Phone Number', 'Gender', 'Date of Birth', 'Institute Name'];
    $headerRowLower = array_map('strtolower', array_map('trim', $headerRow));
    $expectedHeadersLower = array_map('strtolower', $expectedHeaders);
    
    // Check if headers match (flexible matching)
    $headerMatch = true;
    for ($i = 0; $i < min(count($headerRowLower), count($expectedHeadersLower)); $i++) {
        if (empty($headerRowLower[$i]) || strpos($headerRowLower[$i], $expectedHeadersLower[$i]) === false) {
            $headerMatch = false;
            break;
        }
    }
    
    if (!$headerMatch && count($headerRow) < 6) {
        echo json_encode([
            "status" => "error", 
            "message" => "Invalid file format. Please download the template and use the correct column structure."
        ]);
        exit();
    }
    
    $user_id = $_SESSION['user_id'];
    $successCount = 0;
    $errorCount = 0;
    $errors = [];
    $errorRecords = []; // Store failed records with their data and error messages
    $duplicateEmails = [];
    $duplicatePhones = [];
    
    // Process each row
    foreach ($rows as $rowIndex => $row) {
        $lineNumber = $rowIndex + 2; // +2 because we removed header and arrays are 0-indexed
        
        // Skip empty rows
        if (empty(array_filter($row))) {
            continue;
        }
        
        // Extract data (handle both CSV and Excel formats)
        $student_name = isset($row[0]) ? trim($row[0]) : '';
        $email = isset($row[1]) ? trim($row[1]) : '';
        $phone = isset($row[2]) ? trim($row[2]) : '';
        $gender = isset($row[3]) ? strtolower(trim($row[3])) : '';
        
        // Handle date - Excel may return date objects or formatted strings
        $dob_raw = '';
        if (isset($row[4])) {
            $dob_cell = $row[4];
            // If it's a DateTime object (from PhpSpreadsheet), convert to string
            if ($dob_cell instanceof \DateTime) {
                $dob_raw = $dob_cell->format('Y-m-d');
            } elseif (is_numeric($dob_cell) && $dob_cell > 0) {
                // Excel date serial number
                $excelEpoch = mktime(0, 0, 0, 1, 1, 1900);
                $unixTimestamp = $excelEpoch + (($dob_cell - 1) * 86400);
                $dob_raw = date('Y-m-d', $unixTimestamp);
            } else {
                $dob_raw = trim((string)$dob_cell);
            }
        }
        
        $institute_name = isset($row[5]) ? trim($row[5]) : '';
        
        // Normalize date format - handle various Excel/CSV date formats
        $dob = '';
        if (!empty($dob_raw)) {
            $dob = normalizeDate($dob_raw);
        }
        
        $errorMessage = '';
        
        // Validate required fields
        if (empty($student_name)) {
            $errorMessage = "Student Name is required.";
        } elseif (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errorMessage = "Valid Email Address is required.";
        } elseif (empty($phone) || !preg_match('/^\d{10}$/', $phone)) {
            $errorMessage = "Phone Number must be exactly 10 digits.";
        } elseif (empty($gender) || !in_array($gender, ['male', 'female', 'other'])) {
            $errorMessage = "Gender must be 'male', 'female', or 'other'.";
        } elseif (empty($institute_name)) {
            $errorMessage = "Institute Name is required.";
        } elseif (!empty($dob_raw) && empty($dob)) {
            // Date was provided but couldn't be normalized
            $errorMessage = "Date of Birth format is invalid. Please use YYYY-MM-DD format (e.g., 2000-01-15).";
        }
        
        // If validation error, add to error records and continue
        if (!empty($errorMessage)) {
            $errors[] = "Line $lineNumber: $errorMessage";
            $errorRecords[] = [
                'line_number' => $lineNumber,
                'student_name' => $student_name,
                'email' => $email,
                'phone' => $phone,
                'gender' => $gender,
                'dob' => $dob_raw, // Use original date value in error report
                'institute_name' => $institute_name,
                'error_message' => $errorMessage
            ];
            $errorCount++;
            continue;
        }
        
        // Look up institute name in institute_master table to get institute_id
        $stmtInstitute = $pdo->prepare("SELECT institute_id FROM institute_master WHERE institute_name = :institute_name AND is_deleted != 1 LIMIT 1");
        $stmtInstitute->bindParam(':institute_name', $institute_name);
        $stmtInstitute->execute();
        $instituteResult = $stmtInstitute->fetch(PDO::FETCH_ASSOC);
        
        if (!$instituteResult) {
            $errorMessage = "Institute Name '$institute_name' does not exist in the system. Please check the spelling and ensure it matches exactly.";
            $errors[] = "Line $lineNumber: $errorMessage";
            $errorRecords[] = [
                'line_number' => $lineNumber,
                'student_name' => $student_name,
                'email' => $email,
                'phone' => $phone,
                'gender' => $gender,
                'dob' => $dob_raw, // Use original date value in error report
                'institute_name' => $institute_name,
                'error_message' => $errorMessage
            ];
            $errorCount++;
            continue;
        }
        
        $institute_id = $instituteResult['institute_id'];
        
        // Check for duplicate email in current batch
        if (in_array($email, $duplicateEmails)) {
            $errorMessage = "Duplicate email '$email' in the file.";
            $errors[] = "Line $lineNumber: $errorMessage";
            $errorRecords[] = [
                'line_number' => $lineNumber,
                'student_name' => $student_name,
                'email' => $email,
                'phone' => $phone,
                'gender' => $gender,
                'dob' => $dob_raw, // Use original date value in error report
                'institute_name' => $institute_name,
                'error_message' => $errorMessage
            ];
            $errorCount++;
            continue;
        }
        
        // Check for duplicate phone in current batch
        if (in_array($phone, $duplicatePhones)) {
            $errorMessage = "Duplicate phone number '$phone' in the file.";
            $errors[] = "Line $lineNumber: $errorMessage";
            $errorRecords[] = [
                'line_number' => $lineNumber,
                'student_name' => $student_name,
                'email' => $email,
                'phone' => $phone,
                'gender' => $gender,
                'dob' => $dob_raw, // Use original date value in error report
                'institute_name' => $institute_name,
                'error_message' => $errorMessage
            ];
            $errorCount++;
            continue;
        }
        
        // Check for existing email in student_master table (case-insensitive)
        $emailLower = strtolower(trim($email));
        $stmtEmail = $pdo->prepare("SELECT COUNT(*) FROM student_master WHERE LOWER(TRIM(email_id)) = :email");
        $stmtEmail->bindParam(':email', $emailLower);
        $stmtEmail->execute();
        if ($stmtEmail->fetchColumn() > 0) {
            $errorMessage = "Email '$email' already exists in the student records.";
            $errors[] = "Line $lineNumber: $errorMessage";
            $errorRecords[] = [
                'line_number' => $lineNumber,
                'student_name' => $student_name,
                'email' => $email,
                'phone' => $phone,
                'gender' => $gender,
                'dob' => $dob_raw, // Use original date value in error report
                'institute_name' => $institute_name,
                'error_message' => $errorMessage
            ];
            $errorCount++;
            continue;
        }
        
        // Check for existing phone in student_master table
        $phoneTrimmed = trim($phone);
        $stmtPhone = $pdo->prepare("SELECT COUNT(*) FROM student_master WHERE TRIM(phone_no) = :phone");
        $stmtPhone->bindParam(':phone', $phoneTrimmed);
        $stmtPhone->execute();
        if ($stmtPhone->fetchColumn() > 0) {
            $errorMessage = "Phone number '$phone' already exists in the student records.";
            $errors[] = "Line $lineNumber: $errorMessage";
            $errorRecords[] = [
                'line_number' => $lineNumber,
                'student_name' => $student_name,
                'email' => $email,
                'phone' => $phone,
                'gender' => $gender,
                'dob' => $dob_raw, // Use original date value in error report
                'institute_name' => $institute_name,
                'error_message' => $errorMessage
            ];
            $errorCount++;
            continue;
        }
        
        try {
            // Auto-generate random password (8-character alphanumeric)
            $password = substr(str_shuffle('abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'), 0, 8);
            $password_encoded = base64_encode($password);
            
            $is_block = 0;
            $category = 'student';
            
            // Insert into user_master
            $insertUser = $pdo->prepare("
                INSERT INTO user_master (category, email_id, phone_no, password, user_name, created_by, is_block, institute_id)
                VALUES (:category, :email, :phone, :password, :username, :created_by, :is_block, :institute_id)
            ");
            $insertUser->execute([
                ':category' => $category,
                ':email' => $email,
                ':phone' => $phone,
                ':password' => $password_encoded,
                ':username' => $student_name,
                ':created_by' => $user_id,
                ':is_block' => $is_block,
                ':institute_id' => $institute_id
            ]);
            
            $userstudent_id = $pdo->lastInsertId();
            
            // Insert into student_master
            $insertStudent = $pdo->prepare("
                INSERT INTO student_master (user_id, student_name, password, email_id, phone_no, dob, gender, created_by, is_block, institute_id)
                VALUES (:user_id, :student_name, :password, :email, :phone, :dob, :gender, :created_by, :is_block, :institute_id)
            ");
            $insertStudent->execute([
                ':user_id' => $userstudent_id,
                ':student_name' => $student_name,
                ':email' => $email,
                ':phone' => $phone,
                ':password' => $password_encoded,
                ':dob' => $dob ? $dob : null,
                ':gender' => $gender,
                ':created_by' => $user_id,
                ':is_block' => $is_block,
                ':institute_id' => $institute_id
            ]);
            
            // Send welcome email
            sendStudentWelcomeEmail($email, $student_name, $password);
            
            $successCount++;
            $duplicateEmails[] = $email;
            $duplicatePhones[] = $phone;
            
        } catch (PDOException $e) {
            $errorMessage = "Database error - " . $e->getMessage();
            $errors[] = "Line $lineNumber: $errorMessage";
            $errorRecords[] = [
                'line_number' => $lineNumber,
                'student_name' => $student_name,
                'email' => $email,
                'phone' => $phone,
                'gender' => $gender,
                'dob' => $dob_raw, // Use original date value in error report
                'institute_name' => $institute_name,
                'error_message' => $errorMessage
            ];
            $errorCount++;
            error_log("Bulk Upload Error Line $lineNumber: " . $e->getMessage());
        }
    }
    
    // Generate error report file if there are errors
    $errorReportFile = null;
    if ($errorCount > 0 && !empty($errorRecords)) {
        $errorReportFile = generateErrorReport($errorRecords, $headerRow);
    }
    
    // Prepare response - only show summary, all details in CSV file
    $response = [
        "success_count" => $successCount,
        "error_count" => $errorCount
    ];
    
    if ($successCount > 0 && $errorCount == 0) {
        $response["status"] = "success";
        $response["message"] = "Bulk upload completed successfully. Successfully added $successCount student(s).";
    } else if ($successCount > 0 && $errorCount > 0) {
        $response["status"] = "warning";
        $response["message"] = "Bulk upload completed. Successfully added $successCount student(s). $errorCount record(s) failed. Please download the error report for details.";
        if ($errorReportFile) {
            $response["error_report_file"] = $errorReportFile;
        }
    } else {
        $response["status"] = "error";
        $response["message"] = "No students were added. $errorCount record(s) failed. Please download the error report to see all error details.";
        if ($errorReportFile) {
            $response["error_report_file"] = $errorReportFile;
        }
    }
    
    echo json_encode($response);
    
} catch (Exception $e) {
    error_log("Bulk Upload Error: " . $e->getMessage());
    echo json_encode([
        "status" => "error",
        "message" => "An error occurred while processing the file: " . $e->getMessage()
    ]);
}
?>

