<?php
session_start();
ob_start(); // Start output buffering to prevent any output before JSON
header('Content-Type: application/json');
require_once('../includes/connection.php');

try {
    $draw = $_POST['draw'] ?? 1;
    $start = (int)($_POST['start'] ?? 0);
    $length = (int)($_POST['length'] ?? 10);
    $searchValue = $_POST['search']['value'] ?? '';
    $sortField = 'course_id';
    $sortDir = 'ASC';
    $categoryId = $_POST['category_id'] ?? null;

    $userCategory = $_SESSION['category'] ?? null;
    $userId = $_SESSION['user_id'] ?? null;

    /* --------------------------------------------------
       KEEP: Allowable sort fields (UNCHANGED)
    -------------------------------------------------- */
    $allowedSortFields = ['course_title', 'start_date', 'course_fees', 'no_of_licenses'];
    if (!in_array($sortField, $allowedSortFields)) {
        $sortField = 'course_id';
    }

    /* --------------------------------------------------
       NEW: Resolve mentor_id & role ONCE
    -------------------------------------------------- */
    $mentor_id = null;
    $isExpert  = 0;

    if (($userCategory === 'mentor' || $userCategory === 'expert') && !empty($userId)) {
        $stmtRole = $pdo->prepare("
            SELECT mentor_id, industry_expert
            FROM mentor_master
            WHERE user_id = :user_id
        ");
        $stmtRole->execute(['user_id' => $userId]);
        $mentorData = $stmtRole->fetch(PDO::FETCH_ASSOC);

        if ($mentorData) {
            $mentor_id = (int)$mentorData['mentor_id'];
            $isExpert  = (int)$mentorData['industry_expert'];
        } else {
            // Safety: logged in as mentor/expert but no mapping
            ob_end_clean();
            echo json_encode([
                'draw' => intval($draw),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => []
            ]);
            exit;
        }
    }

    /* --------------------------------------------------
       Build search & filter query (MINIMAL CHANGE)
    -------------------------------------------------- */
    $searchQuery = "WHERE c.is_block = 0";
    $params = [];

    if (!empty($categoryId)) {
        $searchQuery .= " AND c.course_category = :category_id";
        $params[':category_id'] = $categoryId;
    }

    // FIXED: mentor/expert filtering uses mentor_id (not user_id)
    if ($userCategory === 'mentor' && $mentor_id && $isExpert === 0) {
        $searchQuery .= " AND FIND_IN_SET(:mentor_id, c.mentor_ids)";
        $params[':mentor_id'] = $mentor_id;
    } elseif ($userCategory === 'expert' && $mentor_id && $isExpert === 1) {
        $searchQuery .= " AND FIND_IN_SET(:mentor_id, c.industry_expert_ids)";
        $params[':mentor_id'] = $mentor_id;
    }

    if (!empty($searchValue)) {
        $searchQuery .= " AND (
            c.course_title LIKE :search_title OR
            c.start_date LIKE :search_start OR
            c.course_fees LIKE :search_fees OR
            c.no_of_licenses LIKE :search_licenses
        )";

        $params[':search_title'] = '%' . $searchValue . '%';
        $params[':search_start'] = '%' . $searchValue . '%';
        $params[':search_fees'] = '%' . $searchValue . '%';
        $params[':search_licenses'] = '%' . $searchValue . '%';
    }

    /* --------------------------------------------------
       Total records (KEEP STRUCTURE)
    -------------------------------------------------- */
    $totalQuery = "SELECT COUNT(*) FROM course_master c WHERE c.is_block != 1";
    $totalParams = [];

    if (!empty($categoryId)) {
        $totalQuery .= " AND c.course_category = :total_category_id";
        $totalParams[':total_category_id'] = $categoryId;
    }

    if ($userCategory === 'mentor' && $mentor_id && $isExpert === 0) {
        $totalQuery .= " AND FIND_IN_SET(:mentor_id, c.mentor_ids)";
        $totalParams[':mentor_id'] = $mentor_id;
    } elseif ($userCategory === 'expert' && $mentor_id && $isExpert === 1) {
        $totalQuery .= " AND FIND_IN_SET(:mentor_id, c.industry_expert_ids)";
        $totalParams[':mentor_id'] = $mentor_id;
    }

    $totalStmt = $pdo->prepare($totalQuery);
    foreach ($totalParams as $k => $v) {
        $totalStmt->bindValue($k, $v);
    }
    $totalStmt->execute();
    $recordsTotal = $totalStmt->fetchColumn();

    /* --------------------------------------------------
       Total filtered records (KEEP STRUCTURE)
    -------------------------------------------------- */
    $filterSql = "
        SELECT COUNT(*)
        FROM course_master c
        LEFT JOIN category_master cm ON c.course_category = cm.category_id
        LEFT JOIN institute_master im ON c.institute_id = im.institute_id
        $searchQuery
    ";

    $filterStmt = $pdo->prepare($filterSql);
    foreach ($params as $k => $v) {
        $filterStmt->bindValue($k, $v);
    }
    $filterStmt->execute();
    $recordsFiltered = $filterStmt->fetchColumn();

    /* --------------------------------------------------
       Data query (KEEP STRUCTURE)
    -------------------------------------------------- */
    $dataSql = "
        SELECT 
            c.*,
            c.expert_approval_status,
            c.expert_approval_comment,
            c.expert_approved_by,
            c.expert_approved_at,
            c.created_by,
            cm.category_name AS course_category,
            cm.category_id,
            im.institute_name,
            u.user_name AS expert_name
        FROM course_master c
        LEFT JOIN category_master cm ON c.course_category = cm.category_id
        LEFT JOIN institute_master im ON c.institute_id = im.institute_id
        LEFT JOIN user_master u ON c.expert_approved_by = u.user_id
        $searchQuery
        ORDER BY $sortField $sortDir
        LIMIT :start, :length
    ";

    $stmt = $pdo->prepare($dataSql);
    foreach ($params as $k => $v) {
        $stmt->bindValue($k, $v);
    }
    $stmt->bindValue(':start', $start, PDO::PARAM_INT);
    $stmt->bindValue(':length', $length, PDO::PARAM_INT);
    $stmt->execute();
    $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

    ob_end_clean();
    echo json_encode([
        'draw' => intval($draw),
        'recordsTotal' => intval($recordsTotal),
        'recordsFiltered' => intval($recordsFiltered),
        'data' => $data
    ]);

} catch (Exception $e) {
    ob_end_clean();
    echo json_encode([
        'draw' => intval($_POST['draw'] ?? 1),
        'recordsTotal' => 0,
        'recordsFiltered' => 0,
        'data' => [],
        'error' => $e->getMessage()
    ]);
}