<?php
session_start();
ob_start(); // Start output buffering to prevent any output before JSON
header('Content-Type: application/json');
require_once('../includes/connection.php');

try {
    $draw = $_POST['draw'] ?? 1;
    $start = (int)($_POST['start'] ?? 0);
    $length = (int)($_POST['length'] ?? 10);
    $searchValue = $_POST['search']['value'] ?? '';
    $sortField =  'course_id';
    $sortDir = 'ASC';
    $categoryId = $_POST['category_id'] ?? null;

    // Get user category and ID from session for filtering
    $userCategory = $_SESSION['category'] ?? null;
    $userId = $_SESSION['user_id'] ?? null;

    // Allowable sort fields
    $allowedSortFields = ['course_title', 'start_date', 'course_fees', 'no_of_licenses'];
    if (!in_array($sortField, $allowedSortFields)) {
        $sortField = 'course_id';
    }

    // Build search & filter query
    $searchQuery = "WHERE c.is_block = 0";
    $params = [];

    // Filter by category if provided
    if (!empty($categoryId)) {
        $searchQuery .= " AND c.course_category = :category_id";
        $params[':category_id'] = $categoryId;
    }

    // Filter by user role - mentors see only their courses, experts see only assigned courses
    if ($userCategory === 'mentor' && !empty($userId)) {
        $searchQuery .= " AND FIND_IN_SET(m.mentor_id, c.mentor_ids)";
    } elseif ($userCategory === 'expert' && !empty($userId)) {
        $searchQuery .= " AND FIND_IN_SET(:expert_user_id, c.industry_expert_ids)";
        $params[':expert_user_id'] = $userId;
    }

    if (!empty($searchValue)) {
        $searchQuery .= " AND (
            c.course_title LIKE :search_title OR
            c.start_date LIKE :search_start OR
            c.course_fees LIKE :search_fees OR
            c.no_of_licenses LIKE :search_licenses
        )";

        $params[':search_title'] = '%' . $searchValue . '%';
        $params[':search_start'] = '%' . $searchValue . '%';
        $params[':search_fees'] = '%' . $searchValue . '%';
        $params[':search_licenses'] = '%' . $searchValue . '%';
    }

    // Total records (without filters) - but respect role-based filtering
    $totalQuery = "SELECT COUNT(*) FROM course_master c";
    $totalParams = [];
    
    // Add JOIN for mentor filtering if user is a mentor
    if ($userCategory === 'mentor' && !empty($userId)) {
        $totalQuery .= " LEFT JOIN mentor_master m ON m.user_id = :total_mentor_join_user_id";
        $totalParams[':total_mentor_join_user_id'] = $userId;
    }
    
    $totalQuery .= " WHERE is_block !=1";
    
    // Add category filter if provided
    if (!empty($categoryId)) {
        $totalQuery .= " AND c.course_category = :total_category_id";
        $totalParams[':total_category_id'] = $categoryId;
    }
    
    // Add role-based filtering
    if ($userCategory === 'mentor' && !empty($userId)) {
        $totalQuery .= " AND FIND_IN_SET(m.mentor_id, c.mentor_ids)";
    } elseif ($userCategory === 'expert' && !empty($userId)) {
        $totalQuery .= " AND FIND_IN_SET(:total_expert_user_id, c.industry_expert_ids)";
        $totalParams[':total_expert_user_id'] = $userId;
    }
    
    if (!empty($totalParams)) {
        $totalStmt = $pdo->prepare($totalQuery);
        foreach ($totalParams as $key => $value) {
            $totalStmt->bindValue($key, $value);
        }
        $totalStmt->execute();
        $recordsTotal = $totalStmt->fetchColumn();
    } else {
        $totalStmt = $pdo->query($totalQuery);
        $recordsTotal = $totalStmt->fetchColumn();
    }

    // Total filtered records
    $filterSql = "
        SELECT COUNT(*) 
        FROM course_master c
        LEFT JOIN category_master cm ON c.course_category = cm.category_id
        LEFT JOIN institute_master im ON c.institute_id = im.institute_id";
    
    // Add JOIN for mentor filtering if user is a mentor
    $filterParams = $params; // Copy params for filter query
    if ($userCategory === 'mentor' && !empty($userId)) {
        $filterSql .= " LEFT JOIN mentor_master m ON m.user_id = :filter_mentor_join_user_id";
        $filterParams[':filter_mentor_join_user_id'] = $userId;
    }
    
    $filterSql .= "
        $searchQuery
    ";
    $filterStmt = $pdo->prepare($filterSql);
    foreach ($filterParams as $key => $value) {
        $filterStmt->bindValue($key, $value);
    }
    $filterStmt->execute();
    $recordsFiltered = $filterStmt->fetchColumn();

    // Data query with filters, sorting, pagination
    $dataSql = "
        SELECT 
            c.*,
            c.expert_approval_status,
            c.expert_approval_comment,
            c.expert_approved_by,
            c.expert_approved_at,
            c.created_by,
            cm.category_name AS course_category,
            cm.category_id,
            im.institute_name,
            u.user_name AS expert_name
        FROM course_master c
        LEFT JOIN category_master cm ON c.course_category = cm.category_id
        LEFT JOIN institute_master im ON c.institute_id = im.institute_id
        LEFT JOIN user_master u ON c.expert_approved_by = u.user_id";
    
    // Add JOIN for mentor filtering if user is a mentor
    $dataParams = $params; // Copy params for data query
    if ($userCategory === 'mentor' && !empty($userId)) {
        $dataSql .= " LEFT JOIN mentor_master m ON m.user_id = :mentor_join_user_id";
        $dataParams[':mentor_join_user_id'] = $userId;
    }
    
    $dataSql .= "
        $searchQuery
        ORDER BY $sortField $sortDir
        LIMIT :start, :length
    ";
    $stmt = $pdo->prepare($dataSql);
    foreach ($dataParams as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':start', $start, PDO::PARAM_INT);
    $stmt->bindValue(':length', $length, PDO::PARAM_INT);
    $stmt->execute();
    $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

    ob_end_clean(); // Clear any output before JSON
    echo json_encode([
        'draw' => intval($draw),
        'recordsTotal' => intval($recordsTotal),
        'recordsFiltered' => intval($recordsFiltered),
        'data' => $data
    ]);
} catch (Exception $e) {
    ob_end_clean(); // Clear any output before JSON
    // Return DataTables-compatible error response
    echo json_encode([
        'draw' => intval($_POST['draw'] ?? 1),
        'recordsTotal' => 0,
        'recordsFiltered' => 0,
        'data' => [],
        'error' => $e->getMessage()
    ]);
}