<?php
// fetch_course_data.php

require_once ('../includes/connection.php'); // Include your database connection

// Get the course ID from the URL (e.g., course.php?id=36)
$draft_id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Check if a valid draft_id was provided
if (!$draft_id) {
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Course ID (id) is required in the URL.']);
    exit();
}

try {
    // 1. Fetch main course data
    $sql = "SELECT * FROM course_draft WHERE draft_id = :draft_id";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':draft_id', $draft_id, PDO::PARAM_INT);
    $stmt->execute();

    $courseData = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$courseData) {
        header('Content-Type: application/json');
        echo json_encode(['error' => 'Course not found with the provided ID.']);
        exit();
    }

    // Initialize arrays and counters for related data to be added to the final response
    $relatedData = [
        'mentors' => [],
        'industry_experts' => [],
        'course_category_name' => null,
        'testimonials_details' => [],
        'proposed_courses_details' => [],
        'faq_details' => [],
        'modules_details' => [],         // Detailed module data with nested submodules and counts
        'total_modules_count' => 0,      // Total count of unique modules
        'total_submodules_count' => 0,   // Total count of unique submodules
        'total_course_videos' => 0,      // Total videos across all modules and submodules in the course
        'total_course_docs' => 0,        // Total documents across all modules and submodules in the course
        'total_modules_only_videos' => 0, // Videos directly in modules (not submodules)
        'total_modules_only_docs' => 0,   // Documents directly in modules (not submodules)
        'total_submodules_only_videos' => 0, // Videos directly in submodules
        'total_submodules_only_docs' => 0,   // Documents directly in submodules
        'course_quizzes_details' => [],  // All quizzes for the entire course
        'skills_gain' => [               // Hardcoded skills_gain array
            "Data Cleaning",
            "Data Visualization",
            "Statistical Analysis",
            "Machine Learning Basics",
            "Python Programming (Intermediate)"
        ],
        'what_youll_learn' => [          // Hardcoded what_youll_learn array
            "Master Pandas for data manipulation.",
            "Create insightful visualizations with Matplotlib.",
            "Understand foundational statistical concepts.",
            "Implement basic machine learning algorithms.",
            "Write efficient Python code for data analysis tasks."
        ]
    ];

    // --- 2. Fetch Related Data (Mentors, Experts, Category, Testimonials, Proposed Courses, FAQs) ---
    // (This part remains largely the same as it's separate from content structure)

    // Fetch Mentor Details from `mentor_master`
    if (!empty($courseData['mentor_ids'])) {
        $mentorIds = explode(',', $courseData['mentor_ids']);
        $placeholders = implode(',', array_fill(0, count($mentorIds), '?'));
        $sql = "SELECT mentor_id, mentor_name FROM mentor_master WHERE mentor_id IN ($placeholders)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($mentorIds);
        $relatedData['mentors'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Fetch Industry Expert Details from `expert_master`
    if (!empty($courseData['industry_expert_ids'])) {
        $expertIds = explode(',', $courseData['industry_expert_ids']);
        $placeholders = implode(',', array_fill(0, count($expertIds), '?'));
        $sql = "SELECT expert_id, expert_name, specialization FROM expert_master WHERE expert_id IN ($placeholders)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($expertIds);
        $relatedData['industry_experts'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Fetch Course Category Name from `category_master`
    if (!empty($courseData['course_category'])) {
        $sql = "SELECT category_name FROM category_master WHERE category_id = :category_id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':category_id', $courseData['course_category'], PDO::PARAM_INT);
        $stmt->execute();
        $category = $stmt->fetch(PDO::FETCH_ASSOC);
        $relatedData['course_category_name'] = $category ? $category['category_name'] : null;
    }

    // Fetch Testimonial Details from `testimonial_master`
    if (!empty($courseData['testimonials'])) {
        $testimonialIds = explode(',', $courseData['testimonials']);
        $placeholders = implode(',', array_fill(0, count($testimonialIds), '?'));
        $sql = "SELECT * FROM testimonial_master WHERE testimonial_id IN ($placeholders)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($testimonialIds);
        $relatedData['testimonials_details'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Fetch Proposed Courses Details from `course_draft`
    if (!empty($courseData['proposed_courses'])) {
        $proposedCourseIds = explode(',', $courseData['proposed_courses']);
        $placeholders = implode(',', array_fill(0, count($proposedCourseIds), '?'));
        $sql = "SELECT draft_id, course_title, course_description, course_img FROM course_draft WHERE draft_id IN ($placeholders)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($proposedCourseIds);
        $relatedData['proposed_courses_details'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Fetch FAQ Details from `faq_master` - Fetching all as requested
    $sql = "SELECT faq_id, faq_question, faq_answer FROM faq_master";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(); // No parameters needed for fetching all
    $relatedData['faq_details'] = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // --- Detailed Fetching and Structuring for Modules, Submodules, Study Materials, and Quizzes ---

    // 1. Get module and submodule IDs linked to this course from course_draft_modules
    $sqlCourseModules = "SELECT module_id, sub_modules FROM course_draft_modules WHERE course_draft_id = :course_draft_id";
    $stmtCourseModules = $pdo->prepare($sqlCourseModules);
    $stmtCourseModules->bindParam(':course_draft_id', $draft_id, PDO::PARAM_INT);
    $stmtCourseModules->execute();
    $courseModulesLinks = $stmtCourseModules->fetchAll(PDO::FETCH_ASSOC);

    $allUniqueModuleIds = [];
    $allUniqueSubmoduleIds = [];
    $moduleSubmoduleMapping = []; // Maps module_id to an array of submodule_ids directly linked under it for this course

    foreach ($courseModulesLinks as $link) {
        $moduleId = (int)$link['module_id'];
        $allUniqueModuleIds[] = $moduleId;
        $moduleSubmoduleMapping[$moduleId] = []; // Initialize for module

        if (!empty($link['sub_modules'])) {
            $submoduleIds = explode(',', $link['sub_modules']);
            foreach ($submoduleIds as $subId) {
                $subId = (int)$subId;
                $allUniqueSubmoduleIds[] = $subId;
                $moduleSubmoduleMapping[$moduleId][] = $subId; // Add submodule to its parent module's list
            }
        }
    }
    $allUniqueModuleIds = array_unique($allUniqueModuleIds);
    $allUniqueSubmoduleIds = array_unique($allUniqueSubmoduleIds);

    $relatedData['total_modules_count'] = count($allUniqueModuleIds);
    $relatedData['total_submodules_count'] = count($allUniqueSubmoduleIds);


    // 2. Fetch all unique study materials details first to avoid redundant queries
    $allStudyMaterialIds = [];
    if(!empty($allUniqueModuleIds)) {
        $placeholders = implode(',', array_fill(0, count($allUniqueModuleIds), '?'));
        $sqlModuleSM = "SELECT study_material FROM module_master WHERE module_id IN ($placeholders)";
        $stmtModuleSM = $pdo->prepare($sqlModuleSM);
        $stmtModuleSM->execute(array_values($allUniqueModuleIds));
        foreach($stmtModuleSM->fetchAll(PDO::FETCH_COLUMN) as $smIds) {
            if(!empty($smIds)) $allStudyMaterialIds = array_merge($allStudyMaterialIds, explode(',', $smIds));
        }
    }
    if(!empty($allUniqueSubmoduleIds)) {
        $placeholders = implode(',', array_fill(0, count($allUniqueSubmoduleIds), '?'));
        $sqlSubmoduleSM = "SELECT study_material FROM submodule_master WHERE submodule_id IN ($placeholders)";
        $stmtSubmoduleSM = $pdo->prepare($sqlSubmoduleSM);
        $stmtSubmoduleSM->execute(array_values($allUniqueSubmoduleIds));
        foreach($stmtSubmoduleSM->fetchAll(PDO::FETCH_COLUMN) as $smIds) {
            if(!empty($smIds)) $allStudyMaterialIds = array_merge($allStudyMaterialIds, explode(',', $smIds));
        }
    }

    $allUniqueStudyMaterialIds = array_unique($allStudyMaterialIds);
    $studyMaterialsDetails = [];
    if (!empty($allUniqueStudyMaterialIds)) {
        $placeholders = implode(',', array_fill(0, count($allUniqueStudyMaterialIds), '?'));
        $sqlSMDetails = "SELECT * FROM study_material WHERE study_material_id IN ($placeholders)";
        $stmtSMDetails = $pdo->prepare($sqlSMDetails);
        $stmtSMDetails->execute(array_values($allUniqueStudyMaterialIds));
        foreach ($stmtSMDetails->fetchAll(PDO::FETCH_ASSOC) as $sm) {
            $studyMaterialsDetails[$sm['study_material_id']] = $sm;
        }
    }

    // 3. Fetch all unique quiz details first
    $allCourseQuizIds = [];
    $sqlAllQuizIds = "SELECT DISTINCT quiz_id FROM course_draft_quiz WHERE course_draft_id = :course_draft_id";
    $stmtAllQuizIds = $pdo->prepare($sqlAllQuizIds);
    $stmtAllQuizIds->bindParam(':course_draft_id', $draft_id, PDO::PARAM_INT);
    $stmtAllQuizIds->execute();
    $allCourseQuizIds = $stmtAllQuizIds->fetchAll(PDO::FETCH_COLUMN, 0);
    $relatedData['total_quizzes'] = count(array_unique($allCourseQuizIds)); // Top-level total quizzes count

    $quizzesDetailsMap = [];
    if (!empty($allCourseQuizIds)) {
        $placeholders = implode(',', array_fill(0, count($allCourseQuizIds), '?'));
        $sqlAllQuizDetails = "SELECT * FROM quiz_master WHERE quiz_id IN ($placeholders)";
        $stmtAllQuizDetails = $pdo->prepare($sqlAllQuizDetails);
        $stmtAllQuizDetails->execute($allCourseQuizIds);
        foreach ($stmtAllQuizDetails->fetchAll(PDO::FETCH_ASSOC) as $quiz) {
            $quizzesDetailsMap[$quiz['quiz_id']] = $quiz;
        }
        $relatedData['course_quizzes_details'] = array_values($quizzesDetailsMap); // Store full details for the course
    }

    // 4. Fetch module master details and build module structure
    $modulesData = []; // This will be the final structured modules_details
    if (!empty($allUniqueModuleIds)) {
        $placeholders = implode(',', array_fill(0, count($allUniqueModuleIds), '?'));
        $sqlModules = "SELECT * FROM module_master WHERE module_id IN ($placeholders)";
        $stmtModules = $pdo->prepare($sqlModules);
        $stmtModules->execute(array_values($allUniqueModuleIds));
        foreach ($stmtModules->fetchAll(PDO::FETCH_ASSOC) as $module) {
            $moduleId = (int)$module['module_id'];
            $module['module_videos'] = 0;
            $module['module_docs'] = 0;
            $module['combined_videos'] = 0;
            $module['combined_docs'] = 0;
            $module['submodules_count'] = 0;
            $module['submodules'] = []; // Initialize nested submodules array
            $module['module_quizzes_count'] = 0; // Quizzes directly attached to module
            $module['combined_quizzes_count'] = 0; // Total quizzes (module + submodules)
            $module['module_quizzes_details'] = [];
            $module['study_materials_details'] = []; // Initialize for study material details

            // Calculate module's own study materials (videos/docs)
            if (!empty($module['study_material'])) {
                $moduleStudyMaterialIds = explode(',', $module['study_material']);
                foreach ($moduleStudyMaterialIds as $smId) {
                    if (isset($studyMaterialsDetails[$smId])) {
                        $material = $studyMaterialsDetails[$smId];
                        // Add the full material details to the module
                        $module['study_materials_details'][] = $material;
                        if (!empty($material['vimeo_link']) && $material['vimeo_link'] !== 'NA') {
                            $module['module_videos']++;
                        }
                        if (!empty($material['actual_file_name']) && $material['actual_file_name'] !== 'NA') {
                            $module['module_docs']++;
                        }
                    }
                }
            }
            $relatedData['total_modules_only_videos'] += $module['module_videos'];
            $relatedData['total_modules_only_docs'] += $module['module_docs'];
            $module['combined_videos'] += $module['module_videos'];
            $module['combined_docs'] += $module['module_docs'];

            // Fetch quizzes directly associated with this module
            $sqlModuleQuizIds = "SELECT DISTINCT quiz_id FROM course_draft_quiz
                                 WHERE course_draft_id = :course_draft_id AND module_id = :module_id AND (submodule_id IS NULL OR submodule_id = 0)";
            $stmtModuleQuizIds = $pdo->prepare($sqlModuleQuizIds);
            $stmtModuleQuizIds->bindParam(':course_draft_id', $draft_id, PDO::PARAM_INT);
            $stmtModuleQuizIds->bindParam(':module_id', $moduleId, PDO::PARAM_INT);
            $stmtModuleQuizIds->execute();
            $moduleQuizIds = $stmtModuleQuizIds->fetchAll(PDO::FETCH_COLUMN, 0);
            $module['module_quizzes_count'] = count($moduleQuizIds);
            $module['combined_quizzes_count'] = $module['module_quizzes_count']; // Initialize combined count with module's own quizzes
            foreach ($moduleQuizIds as $quizId) {
                if (isset($quizzesDetailsMap[$quizId])) {
                    $module['module_quizzes_details'][] = $quizzesDetailsMap[$quizId];
                }
            }
            $modulesData[$moduleId] = $module;
        }
    }

    // 5. Fetch submodule master details and prepare them for nesting
    $flatSubmodulesData = []; // A temporary flat map of all submodules by ID
    if (!empty($allUniqueSubmoduleIds)) {
        $placeholders = implode(',', array_fill(0, count($allUniqueSubmoduleIds), '?'));
        $sqlSubmodules = "SELECT * FROM submodule_master WHERE submodule_id IN ($placeholders)";
        $stmtSubmodules = $pdo->prepare($sqlSubmodules);
        $stmtSubmodules->execute(array_values($allUniqueSubmoduleIds));
        foreach ($stmtSubmodules->fetchAll(PDO::FETCH_ASSOC) as $submodule) {
            $submoduleId = (int)$submodule['submodule_id'];
            $submodule['submodule_videos'] = 0;
            $submodule['submodule_docs'] = 0;
            $submodule['submodule_quizzes_count'] = 0;
            $submodule['submodule_quizzes_details'] = [];
            $submodule['study_materials_details'] = []; // Initialize for study material details

            // Count videos/docs for the submodule itself and populate details
            if (!empty($submodule['study_material'])) {
                $submoduleStudyMaterialIds = explode(',', $submodule['study_material']);
                foreach ($submoduleStudyMaterialIds as $smId) {
                    if (isset($studyMaterialsDetails[$smId])) {
                        $material = $studyMaterialsDetails[$smId];
                        $submodule['study_materials_details'][] = $material; // Add study material details here
                        if (!empty($material['vimeo_link']) && $material['vimeo_link'] !== 'NA') {
                            $submodule['submodule_videos']++;
                        }
                        if (!empty($material['actual_file_name']) && $material['actual_file_name'] !== 'NA') {
                            $submodule['submodule_docs']++;
                        }
                    }
                }
            }
            $relatedData['total_submodules_only_videos'] += $submodule['submodule_videos'];
            $relatedData['total_submodules_only_docs'] += $submodule['submodule_docs'];

            // Fetch quizzes for this specific submodule for the current course draft
            $sqlSubmoduleQuizIds = "SELECT DISTINCT quiz_id FROM course_draft_quiz
                                    WHERE course_draft_id = :course_draft_id
                                    AND submodule_id = :submodule_id";
            $stmtSubmoduleQuizIds = $pdo->prepare($sqlSubmoduleQuizIds);
            $stmtSubmoduleQuizIds->bindParam(':course_draft_id', $draft_id, PDO::PARAM_INT);
            $stmtSubmoduleQuizIds->bindParam(':submodule_id', $submoduleId, PDO::PARAM_INT);
            $stmtSubmoduleQuizIds->execute();
            $submoduleQuizIds = $stmtSubmoduleQuizIds->fetchAll(PDO::FETCH_COLUMN, 0);

            $submodule['submodule_quizzes_count'] = count($submoduleQuizIds);
            foreach ($submoduleQuizIds as $quizId) {
                if (isset($quizzesDetailsMap[$quizId])) {
                    $submodule['submodule_quizzes_details'][] = $quizzesDetailsMap[$quizId];
                }
            }
            $flatSubmodulesData[$submoduleId] = $submodule;
        }
    }
    // Set the top-level submodules_details (this is a flat list of all submodules found in the course)
    $relatedData['submodules_details'] = array_values($flatSubmodulesData);


    // 6. Nest submodules under their respective modules based on course_draft_modules mapping
    // Iterate over modules and add their specific submodules
    foreach ($modulesData as $moduleId => &$module) {
        if (isset($moduleSubmoduleMapping[$moduleId])) {
            foreach ($moduleSubmoduleMapping[$moduleId] as $subId) {
                if (isset($flatSubmodulesData[$subId])) {
                    $submodule = $flatSubmodulesData[$subId];
                    // Explicitly set the module_id in the context of this course's structure
                    $submodule['module_id_in_course_context'] = $moduleId;
                    $module['submodules'][] = $submodule;
                    $module['submodules_count']++;
                    $module['combined_videos'] += $submodule['submodule_videos'];
                    $module['combined_docs'] += $submodule['submodule_docs'];
                    $module['combined_quizzes_count'] += $submodule['submodule_quizzes_count']; // Add submodule quizzes to module's combined count
                }
            }
        }
    }
    unset($module); // Break the reference to the last element

    // Add module data to final relatedData array in desired order (e.g., by module_id)
    usort($modulesData, function($a, $b) {
        return $a['module_id'] - $b['module_id'];
    });
    $relatedData['modules_details'] = array_values($modulesData);


    // Calculate total course videos and docs from modules_details for clarity
    // (This ensures consistency after nesting logic)
    $totalCourseVideos = 0;
    $totalCourseDocs = 0;
    foreach ($relatedData['modules_details'] as $module) {
        $totalCourseVideos += $module['combined_videos'];
        $totalCourseDocs += $module['combined_docs'];
    }
    $relatedData['total_course_videos'] = $totalCourseVideos;
    $relatedData['total_course_docs'] = $totalCourseDocs;


    // Combine main course data with all related data
    $finalResponse = array_merge($courseData, $relatedData);

    header('Content-Type: application/json');
    echo json_encode($finalResponse);

} catch (PDOException $e) {
    // Log the error for debugging purposes
    error_log("Database error in fetch_course_data.php: " . $e->getMessage());
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
}

?>