<?php
header('Content-Type: application/json');
session_start(); // Ensure session is started to access $_SESSION['institute_id'] if needed

require_once('../includes/connection.php'); // Adjust path as per your project structure

// Ensure error reporting is configured for development (remove in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Input validation: Get draft_id and module_id from GET parameters
$draftId = isset($_GET['draft_id']) ? intval($_GET['draft_id']) : null;
$moduleId = isset($_GET['module_id']) ? intval($_GET['module_id']) : null;

// Validate that both IDs are provided
if (!$draftId || !$moduleId) {
    http_response_code(400); // Bad Request
    echo json_encode(['success' => false, 'message' => 'Both draft_id and module_id are required.']);
    exit;
}

try {
    // 1. Fetch the comma-separated submodule IDs from the course_draft_modules table
    //    for the specific draft_id and module_id.
    $sqlFetchLinkedSubmodules = "
        SELECT
            sub_modules
        FROM
            course_draft_modules
        WHERE
            course_draft_id = :draft_id AND module_id = :module_id;
    ";
    $stmtFetchLinkedSubmodules = $pdo->prepare($sqlFetchLinkedSubmodules);
    $stmtFetchLinkedSubmodules->bindParam(':draft_id', $draftId, PDO::PARAM_INT);
    $stmtFetchLinkedSubmodules->bindParam(':module_id', $moduleId, PDO::PARAM_INT);
    $stmtFetchLinkedSubmodules->execute();
    $result = $stmtFetchLinkedSubmodules->fetch(PDO::FETCH_ASSOC);

    $submodules = []; // Initialize an empty array to store the fetched submodules

    // Check if a result was found and if sub_modules column is not empty
    if ($result && !empty($result['sub_modules'])) {
        $linkedSubmoduleIdsCsv = $result['sub_modules'];
        // Split the CSV string into an array of integer IDs, filtering out any empty values
        $submoduleIds = array_map('intval', array_filter(explode(',', $linkedSubmoduleIdsCsv)));

        // If there are valid submodule IDs, proceed to fetch their details
        if (!empty($submoduleIds)) {
            // Create a string of placeholders for the SQL IN clause (e.g., ?, ?, ?)
            $placeholders = implode(',', array_fill(0, count($submoduleIds), '?'));

            // 2. Fetch the details (id and name) for these specific submodules from submodule_master.
            //    Using FIELD() in ORDER BY to maintain the original order from the CSV list.
            $sqlSubmoduleDetails = "
                SELECT
                    sm.submodule_id AS id,
                    sm.submodule_name AS name
                FROM
                    submodule_master sm
                WHERE
                    sm.submodule_id IN ($placeholders)
                ORDER BY FIELD(sm.submodule_id, $placeholders);
            ";
            $stmtSubmoduleDetails = $pdo->prepare($sqlSubmoduleDetails);
            // Bind all parameters: first set for the IN clause, second set for the FIELD clause
            $stmtSubmoduleDetails->execute(array_merge($submoduleIds, $submoduleIds));
            $submodules = $stmtSubmoduleDetails->fetchAll(PDO::FETCH_ASSOC);
        }
    }

    // Send a successful JSON response with the list of submodules
    http_response_code(200);
    echo json_encode(['success' => true, 'submodules' => $submodules]);

} catch (PDOException $e) {
    // Handle database errors
    http_response_code(500); // Internal Server Error
    error_log("Database error in fetch_draft_submodules.php: " . $e->getMessage()); // Log error for debugging
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
} catch (Exception $e) {
    // Handle any other unexpected errors
    http_response_code(500); // Internal Server Error
    error_log("General error in fetch_draft_submodules.php: " . $e->getMessage()); // Log error
    echo json_encode(['success' => false, 'message' => 'An unexpected server error occurred.']);
}

// Ensure all output buffers are flushed
ob_end_flush();
?>