<?php
require_once('../includes/connection.php');
header('Content-Type: application/json');

$user_id = $_SESSION['user_id'] ?? null;
$institute_id = $_SESSION['institute_id'] ?? null;

try {
    // Support both JSON and form-data
    if (strpos($_SERVER["CONTENT_TYPE"], "application/json") !== false) {
        $data = json_decode(file_get_contents("php://input"), true);
    } else {
        $data = $_POST;
    }

    if (!$data) {
        throw new Exception("Invalid input.");
    }

    // Extract fields
    $course_id = $data['course_id'] ?? null;
    $course_title = $data['course_title'] ?? $data['course_name'] ?? null;
    $course_description = $data['course_description'] ?? $data['description'] ?? null;
    $category = $data['category'] ?? $data['course_category'] ?? null;
    $level = $data['level'] ?? null;
    $language = $data['language'] ?? null;
    $licenses = $data['licenses'] ?? null;
    $course_fees = $data['course_fees'] ?? null;
    $hours = $data['hours'] ?? null;
    $start_date = $data['start_date'] ?? null;
    $end_date = $data['end_date'] ?? null;
    $mentor = isset($data['mentor']) && is_array($data['mentor']) ? implode(',', $data['mentor']) : $data['mentor'] ?? null;
    $industry_expert = isset($data['industry_expert']) && is_array($data['industry_expert']) ? implode(',', $data['industry_expert']) : $data['industry_expert'] ?? null;
    $testimonials = isset($data['testimonials']) && is_array($data['testimonials']) ? implode(',', $data['testimonials']) : $data['testimonials'] ?? null;
    $proposed_courses = isset($data['proposed_courses']) && is_array($data['proposed_courses']) ? implode(',', $data['proposed_courses']) : $data['proposed_courses'] ?? null;
    $faqs = isset($data['faqs']) && is_array($data['faqs']) ? implode(',', $data['faqs']) : $data['faqs'] ?? null;
    $skills = $data['skills'] ?? null;
    $learnings = $data['learnings'] ?? null;

    if (!$course_id || !$course_title || !$licenses || !$category || !$start_date || !$end_date || !$course_description) {
        throw new Exception("Missing required fields.");
    }

    // Get current course image
    $stmt = $pdo->prepare("SELECT course_img FROM course_master WHERE course_id = :course_id");
    $stmt->execute([':course_id' => $course_id]);
    $existingImg = $stmt->fetchColumn();

    $courseImg = $existingImg;

    // Handle image upload
    if (isset($_FILES['course_img']) && $_FILES['course_img']['error'] === UPLOAD_ERR_OK) {
        $fileName = $_FILES['course_img']['name'];
        $tmpPath = $_FILES['course_img']['tmp_name'];
        $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
        $encryptedFileName = hash('sha256', uniqid('', true)) . '.' . $fileExt;

        // Get folder name
        $stmt = $pdo->prepare("SELECT folder_name FROM institute_master WHERE institute_id = :institute_id");
        $stmt->execute([':institute_id' => $institute_id]);
        $folderName = $stmt->fetchColumn();

        if (!$folderName) {
            throw new Exception("Institute folder not found. Please log in again and try.");
        }

        $uploadDir = '../institutes/' . $folderName . '/course_images/';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }

        $targetPath = $uploadDir . $encryptedFileName;

        if (!move_uploaded_file($tmpPath, $targetPath)) {
            throw new Exception("Failed to upload course image.");
        }

        // Delete old image
        if (!empty($existingImg) && file_exists($uploadDir . $existingImg)) {
            unlink($uploadDir . $existingImg);
        }

        $courseImg = $encryptedFileName;
    }
    
    
        // Check if course title already exists (unique per institute, except current course)
        $check = $pdo->prepare("
            SELECT COUNT(*) FROM course_master
            WHERE course_title = :course_title 
              AND institute_id = :institute_id
              AND course_id != :course_id
        ");
        $check->execute([
            ':course_title' => $course_title,
            ':institute_id' => $institute_id,
            ':course_id' => $course_id
        ]);
        
        if ($check->fetchColumn() > 0) {
            throw new Exception("Course title already exists. Please choose a different name.");
        }

    

    // Update query
    $stmt = $pdo->prepare("UPDATE course_master SET
        course_title = :course_title,
        course_description = :course_description,
        course_category = :category,
        level = :level,
        hours = :hours,
        language = :language,
        no_of_licenses = :licenses,
        course_fees = :course_fees,
        start_date = :start_date,
        end_date = :end_date,
        mentor_ids = :mentor,
        industry_expert_ids = :industry_expert,
        testimonials = :testimonials,
        proposed_courses = :proposed_courses,
        faq_ids = :faqs,
        skills = :skills,
        learnings = :learnings,
        course_img = :course_img
        WHERE course_id = :course_id
    ");

    $stmt->execute([
        ':course_title' => $course_title,
        ':course_description' => $course_description,
        ':category' => $category,
        ':level' => $level,
        ':hours' => $hours,
        ':language' => $language,
        ':licenses' => $licenses,
        ':course_fees' => $course_fees,
        ':start_date' => $start_date,
        ':end_date' => $end_date,
        ':mentor' => $mentor,
        ':industry_expert' => $industry_expert,
        ':testimonials' => $testimonials,
        ':proposed_courses' => $proposed_courses,
        ':faqs' => $faqs,
        ':skills' => $skills,
        ':learnings' => $learnings,        
        ':course_img' => $courseImg,
        ':course_id' => $course_id
    ]);

    echo json_encode([
        'status' => 'success',
        'message' => 'Course updated successfully.',
        'course_id' => $course_id
    ]);

} catch (Exception $e) {
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
}
?>