<?php
require_once('../includes/connection.php');
header('Content-Type: application/json');

// Validate course_id
if (!isset($_GET['course_id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Missing course_id']);
    exit;
}

$course_id = intval($_GET['course_id']);

try {
    $institute_id = $_SESSION['institute_id'] ?? null;
    $user_id = $_SESSION['user_id'] ?? null;

    if (!$institute_id) {
        echo json_encode(['status' => 'error', 'message' => 'Institute ID not found in session.']);
        exit;
    }

    if (!$user_id) {
        echo json_encode(['status' => 'error', 'message' => 'User ID not found in session.']);
        exit;
    }

    // Get folder name for current institute
    $stmtFolder = $pdo->prepare("SELECT folder_name FROM institute_master WHERE institute_id = :institute_id");
    $stmtFolder->execute([':institute_id' => $institute_id]);
    $currentFolder = $stmtFolder->fetchColumn();

    if (!$currentFolder) {
        echo json_encode(["status" => "error", "message" => "Institute folder not found."]);
        exit;
    }

    // Get superadmin folder
    $stmtSuperFolder = $pdo->prepare("SELECT folder_name FROM institute_master WHERE institute_id = 1");
    $stmtSuperFolder->execute();
    $superadminFolder = $stmtSuperFolder->fetchColumn();

    // Fetch course info
    $courseStmt = $pdo->prepare("
        SELECT course_title, skills, course_description, course_img, mentor_ids, learnings, is_superadmin_course, hours, language, level
        FROM course_master
        WHERE course_id = :id
    ");
    $courseStmt->bindParam(':id', $course_id);
    $courseStmt->execute();
    $course = $courseStmt->fetch(PDO::FETCH_ASSOC);

    if (!$course) {
        echo json_encode(['status' => 'error', 'message' => 'Course not found']);
        exit;
    }

    // Image path
    $imageName = $course['course_img'];
    if (!empty($imageName)) {
        $folderToUse = ($course['is_superadmin_course'] ?? 0) == 1 ? $superadminFolder : $currentFolder;
        $course['course_img'] = "institutes/{$folderToUse}/course_images/{$imageName}";
    }

    // Fetch mentor names
    $mentorNames = [];
    $industryExperts = [];
    if (!empty($course['mentor_ids'])) {
        $mentorIdsArray = array_filter(explode(',', $course['mentor_ids']), 'is_numeric');
        if (!empty($mentorIdsArray)) {
            $placeholders = implode(',', array_fill(0, count($mentorIdsArray), '?'));
            $mentorStmt = $pdo->prepare("
                SELECT mentor_id, mentor_name, industry_expert 
                FROM mentor_master 
                WHERE mentor_id IN ($placeholders)
            ");
            $mentorStmt->execute($mentorIdsArray);
            $mentors = $mentorStmt->fetchAll(PDO::FETCH_ASSOC);
            foreach ($mentors as $mentor) {
                $mentorData = [
                    'mentor_id' => $mentor['mentor_id'],
                    'mentor_name' => $mentor['mentor_name']
                ];
                $mentorNames[] = $mentorData;
                if ((int)$mentor['industry_expert'] === 1) {
                    $industryExperts[] = $mentorData;
                }
            }
        }
    }

    // Fetch modules
    $moduleStmt = $pdo->prepare("
        SELECT mm.module_id, mm.module_name, mm.description, mm.study_material
        FROM course_module cm
        INNER JOIN module_master mm ON cm.module_id = mm.module_id
        WHERE cm.course_id = :course_id
        ORDER BY cm.sequence ASC
    ");
    $moduleStmt->bindParam(':course_id', $course_id);
    $moduleStmt->execute();
    $modules = $moduleStmt->fetchAll(PDO::FETCH_ASSOC);

    // Fetch submodules
    $submoduleStmt = $pdo->prepare("
        SELECT sm.submodule_id, sm.submodule_name, sm.description, sm.study_material
        FROM course_submodule cs
        INNER JOIN submodule_master sm ON cs.submodule_id = sm.submodule_id
        WHERE cs.course_id = :course_id
    ");
    $submoduleStmt->bindParam(':course_id', $course_id);
    $submoduleStmt->execute();
    $submodules = $submoduleStmt->fetchAll(PDO::FETCH_ASSOC);

    // Helper to count materials
    function countStudyMaterials($pdo, $csv) {
        $video = 0; $file = 0;
        $ids = array_filter(explode(',', $csv));
        foreach ($ids as $id) {
            $stmt = $pdo->prepare("SELECT doc_type FROM study_material WHERE study_material_id = :id");
            $stmt->execute([':id' => $id]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($row) {
                if ($row['doc_type'] === 'video') $video++;
                elseif ($row['doc_type'] === 'file') $file++;
            }
        }
        return [$video, $file];
    }

    // Count of study materials (videos, files)
    $modulesWithCounts = [];
    $totalModules = 0;
    $totalVideos = 0;
    $totalFiles = 0;

    foreach ($modules as $module) {
        [$videoCount, $fileCount] = countStudyMaterials($pdo, $module['study_material']);
        $modulesWithCounts[] = [
            'name' => $module['module_name'],
            'type' => 'module',
            'video_count' => $videoCount,
            'file_count' => $fileCount
        ];
        $totalModules++;
        $totalVideos += $videoCount;
        $totalFiles += $fileCount;
    }

    foreach ($submodules as $submodule) {
        [$videoCount, $fileCount] = countStudyMaterials($pdo, $submodule['study_material']);
        $modulesWithCounts[] = [
            'name' => $submodule['submodule_name'],
            'type' => 'submodule',
            'video_count' => $videoCount,
            'file_count' => $fileCount
        ];
        $totalVideos += $videoCount;
        $totalFiles += $fileCount;
    }

    // ✅ Check user progress to set button_name
    $progressStmt = $pdo->prepare("
        SELECT COUNT(*) FROM user_course_progress
        WHERE course_id = :course_id AND user_id = :user_id
    ");
    $progressStmt->execute([
        ':course_id' => $course_id,
        ':user_id' => $user_id
    ]);
    $hasProgress = $progressStmt->fetchColumn() > 0;

    $buttonName = $hasProgress ? 'Resume Course' : 'Start Course';

    // Final response
    $response = [
        'status' => 'success',
        'data' => [
            'course_title' => $course['course_title'],
            'course_description' => $course['course_description'],
            'course_img' => $course['course_img'],
            'skills' => $course['skills'],
            'learnings' => $course['learnings'],
            'hours' => $course['hours'],
            'level' => $course['level'],
            'language' => $course['language'],
            'mentors' => $mentorNames,
            'modules' => $modules,
            'submodules' => $submodules,
            'industry_experts' => $industryExperts,
            'module_count' => $totalModules,
            'video_count' => $totalVideos,
            'file_count' => $totalFiles,
            'modules_with_counts' => $modulesWithCounts,
            'button_name' => $buttonName
        ]
    ];

    echo json_encode($response);

} catch (PDOException $e) {
    echo json_encode([
        'status' => 'error',
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
