<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <title>Module and Study Video Manager</title>
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" />
  <style>
    .study-row {
      display: flex;
      gap: 10px;
      align-items: center;
      margin-bottom: 10px;
    }
    .study-row input[type="text"] {
      flex: 2;
    }
    .study-row input[type="file"] {
      flex: 3;
    }
  </style>
</head>
<body class="p-4">

  <!-- Button to open modal -->
  <button class="btn btn-primary mb-3" data-bs-toggle="modal" data-bs-target="#existingModulesModal">
    Select Existing Modules
  </button>

  <!-- Table of selected/created modules -->
  <h5>Existing Modules</h5>
  <table class="table table-bordered" id="moduleTable">
    <thead>
      <tr>
        <th>Select</th>
        <th>Module Name</th>
      </tr>
    </thead>
    <tbody id="moduleTableBody">
      <!-- Filled dynamically -->
    </tbody>
  </table>

  <!-- Button to create new module -->
  <button class="btn btn-success mb-3" onclick="addNewModule()">+ Create New Module</button>

  <!-- Container for new modules -->
  <div id="newModulesContainer"></div>

  <!-- Modal for existing modules -->
  <div class="modal fade" id="existingModulesModal" tabindex="-1" aria-labelledby="existingModulesModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-scrollable">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title">Select Existing Modules</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <ul class="list-group" id="existingModulesList">
            <!-- Populated dynamically -->
          </ul>
        </div>
        <div class="modal-footer">
          <button class="btn btn-primary" onclick="addSelectedModules()" data-bs-dismiss="modal">Add Selected Modules</button>
        </div>
      </div>
    </div>
  </div>

  <!-- JS -->
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>

<!-- [no changes to head or body content above this point] -->

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>

<script>

//const course_id = $('#course_id').val();
const course_id = 1;

function loadExistingModules() {
  $.ajax({
    url: './cr/get_modules_api.php',
    type: 'POST',
    dataType: 'json',
    success: function (data) {
      const list = $('#existingModulesList');
      list.empty();

      if (data.status !== 'success' || !Array.isArray(data.data)) {
        list.append('<li class="list-group-item text-danger">Invalid response format</li>');
        return;
      }

      const addedModuleIds = new Set();
      $('#newModulesContainer .card').each(function () {
        const id = $(this).attr('id');
        const match = id && id.match(/^module_(\d+)$/);
        if (match) addedModuleIds.add(match[1]);
      });

      data.data.forEach(function (mod) {
        const isChecked = addedModuleIds.has(mod.module_id.toString()) ? 'checked' : '';
        const li = `
          <li class="list-group-item">
            <input type="checkbox" class="form-check-input me-2" value="${mod.module_id}" data-module-name="${mod.module_name}" ${isChecked}>
            ${mod.module_name}
          </li>`;
        list.append(li);
      });
    },
    error: function () {
      $('#existingModulesList').html('<li class="list-group-item text-danger">Failed to load modules.</li>');
    }
  });
}

function refreshModulesUI(course_id) {
  $.ajax({
    url: './cr/get_initial_data.php',
    method: 'GET',
    data: { course_id: course_id }, // <-- Pass course_id here
    dataType: 'json',
    success: function (res) {
      if (res.status === 'success') {
        $('#moduleTableBody').empty();
        $('#newModulesContainer').empty();
        renderModuleCards(res.data);
        renderModuleTable(res.data);
      }
    }
  });
}


window.onload = function () {
  refreshModulesUI(course_id);

  $('#existingModulesModal').on('shown.bs.modal', function () {
    loadExistingModules();
  });

  let moduleCounter = 1;

  window.addNewModule = function () {
    const moduleId = `newModule${moduleCounter++}`;
    const container = document.getElementById('newModulesContainer');

    const card = document.createElement('div');
    card.className = 'card mb-3';
    card.id = moduleId;
    card.innerHTML = `
      <div class="card-body">
      

    <div class="col-md-6">
        <input type="text" class="form-control study-title" value="Study - 1" readonly="">
    </div>
    
    <div class="d-flex align-items-center mb-3">
        <input type="text" id="${moduleId}-title" class="form-control me-2" placeholder="Enter Module Name" />
        <button class="btn btn-outline-primary" onclick="submitStudies('${moduleId}', 0)">➕ Module</button>
    </div>
    <div id="${moduleId}-study-rows"></div>
        <button class="btn btn-secondary btn-sm" onclick="addStudyRow('${moduleId}')">+ Add Study</button>
    </div>
    `;
    container.appendChild(card);
  };

  window.submitStudies = function (moduleDomId, moduleDbId) {
    const titleInput = document.getElementById(`${moduleDomId}-title`);
    const title = titleInput.value.trim();

    if (!title) {
      alert("Please enter a module name.");
      return;
    }

    $.ajax({
      type: "POST",
      url: "./cr/add_module_and_assign_api.php",
      data: { module_name: title },
      dataType: "json",
      success: function (res) {
        if (res.status === "success") {
          alert("Module added successfully.");
          const saveBtn = document.querySelector(`#${moduleDomId} .btn-outline-primary`);
          saveBtn.setAttribute("onclick", `submitStudies('${moduleDomId}', ${res.module_id})`);
          refreshModulesUI(course_id);
        } else {
          alert("Failed to add module.");
        }
      },
      error: function () {
        alert("AJAX error occurred.");
      }
    });
  };

  window.addStudyRow = function (moduleId) {
    const container = document.getElementById(`${moduleId}-study-rows`);
    const row = document.createElement("div");
    row.className = "study-row";
    row.innerHTML = `
      <input type="text" class="form-control study-title" placeholder="Study Title" />
      <input type="file" class="form-control study-file" />
      <button class="btn btn-info btn-sm" onclick="uploadSingleStudy(this, '${moduleId}')">Upload</button>
      <span class="upload-status ms-2"></span>
      <button class="btn btn-danger btn-sm" onclick="deleteVideoFromVimeo(this)">🗑</button>
    `;
    container.appendChild(row);
  };

  window.uploadSingleStudy = function (btn, moduleDomId) {
    const row = btn.parentElement;
    const titleInput = row.querySelector(".study-title");
    const fileInput = row.querySelector(".study-file");
    const statusSpan = row.querySelector(".upload-status");

    const title = titleInput.value.trim();
    const file = fileInput.files[0];

    if (!title || !file) {
      alert("Please provide both title and file.");
      return;
    }

    const moduleDbId = getModuleDbIdFromDomId(moduleDomId);
    if (!moduleDbId) {
      alert("Module ID not found. Please save the module first.");
      return;
    }

    const formData = new FormData();
    formData.append("action", "upload_single_study");
    formData.append("module_id", moduleDbId);
    formData.append("title", title);
    formData.append("file", file);

    statusSpan.textContent = "⏳ Uploading...";

    $.ajax({
      url: "./cr/upload_file.php",
      method: "POST",
      data: formData,
      processData: false,
      contentType: false,
      success: function (res) {
        if (res.status === "success") {
          statusSpan.textContent = "✅ Uploaded";
          if (res.video_id) {
            row.setAttribute("data-video-id", res.video_id);
          }
        } else {
          statusSpan.textContent = "❌ Failed";
        }
      },
      error: function () {
        statusSpan.textContent = "❌ Error";
      },
    });
  };

  function getModuleDbIdFromDomId(moduleDomId) {
    const card = document.getElementById(moduleDomId);
    const saveBtn = card.querySelector(".btn-outline-primary");
    if (!saveBtn) return null;
    const onclickAttr = saveBtn.getAttribute("onclick");
    const match = onclickAttr && onclickAttr.match(/submitStudies\('\w+',\s*(\d+)\)/);
    return match ? match[1] : null;
  }

  window.addSelectedModules = function () {
    const selected = $('#existingModulesList input:checked');
    if (selected.length === 0) {
      alert('Please select at least one module.');
      return;
    }

    const moduleIds = [];
    const moduleNames = {};

    selected.each(function () {
      const moduleId = $(this).val();
      const moduleName = $(this).data('module-name');
      moduleIds.push(moduleId);
      moduleNames[moduleId] = moduleName;
    });

    $.ajax({
      url: './cr/add_course_module_api.php',
      type: 'POST',
      data: JSON.stringify({ module_ids: moduleIds }),
      contentType: 'application/json',
      success: function (response) {
        if (response.status === 'success') {
          response.inserted.forEach(function (moduleId) {
            const moduleName = moduleNames[moduleId];
            renderModuleCard({
              module_id: moduleId,
              module_name: moduleName,
              study_materials: [],
              linked_study_ids: []
            });
          });

          refreshModulesUI(course_id);
          alert(response.message);
        } else {
          alert(response.message || 'Something went wrong!');
        }
      },
      error: function (xhr) {
        console.error(xhr.responseText);
        alert('Error contacting API.');
      }
    });
  };

  window.deleteVideoFromVimeo = function (btn) {
    const row = btn.closest('.study-row');
    const videoId = row.getAttribute('data-video-id');

    if (!videoId) {
      row.remove();
      return;
    }

    if (!confirm("Are you sure you want to delete this video from Vimeo?")) return;

    $.ajax({
      url: './cr/delete_vimeo_video.php',
      type: 'POST',
      data: { video_id: videoId },
      dataType: 'json',
      success: function (response) {
        if (response.success) {
          row.remove();
          alert('Video deleted successfully.');
        } else {
          alert('Failed to delete video: ' + response.message);
        }
      },
      error: function (xhr, status, error) {
        alert('AJAX error: ' + error);
      }
    });
  };
};

// Table rows for module checklist
function renderModuleTable(modules) {
  const tableBody = document.getElementById("moduleTableBody");
  tableBody.innerHTML = '';

  modules.forEach(function (mod) {
    const tr = document.createElement("tr");
    tr.innerHTML = `
      <td><input type="checkbox" data-module-id="${mod.module_id}"></td>
      <td>${mod.module_name}</td>
    `;
    tableBody.appendChild(tr);
  });
}

// Render full module card with study material rows
function renderModuleCards(modules) {
  modules.forEach(renderModuleCard);
}

// Card builder
function renderModuleCard(mod) {
  const moduleId = mod.module_id;
  const moduleName = mod.module_name;
  const uniqueId = 'module_' + moduleId;

  if (document.getElementById(uniqueId)) return;

  let studyRowsHTML = '';
  mod.study_materials.forEach(material => {
    const title = material.title;
    const vimeoLink = material.vimeo_link || '#';
    const fileName = material.actual_file_name;
    const videoId = material.study_material_id;

    studyRowsHTML += `
      <div class="study-row" data-video-id="${videoId}">
        <input type="text" class="form-control study-title" value="${title}" readonly />
        <a href="${vimeoLink}" target="_blank">${fileName}</a>
        <button class="btn btn-danger btn-sm" onclick="deleteVideoFromVimeo(this)">🗑</button>
      </div>
    `;
  });

  const linkedIds = mod.linked_study_ids ? mod.linked_study_ids.join(',') : '';

  const moduleHTML = `
    <div class="card my-3" id="${uniqueId}" data-linked-study-ids="${linkedIds}">
      <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">${moduleName}</h5>
        <button class="btn btn-sm btn-danger remove-module-btn" data-module-id="${moduleId}" data-card-id="${uniqueId}">Remove</button>
      </div>
      <div class="card-body">
        <div id="${uniqueId}-study-rows">${studyRowsHTML}</div>
        <button class="btn btn-secondary btn-sm mb-2" onclick="addStudyRow('${uniqueId}')">+ Add Study</button>
        <button class="btn btn-outline-primary" onclick="submitStudies('${uniqueId}', ${moduleId})">Save Studies</button>
      </div>
    </div>
  `;

  $('#newModulesContainer').append(moduleHTML);
}

$(document).on('click', '.remove-module-btn', function () {
  const moduleId = $(this).data('module-id');
  const cardId = $(this).data('card-id');

  if (!confirm("Are you sure you want to remove this module from course_module?")) return;

  $.ajax({
    url: './cr/delete_course_module_api.php',
    method: 'POST',
    data: { module_id: moduleId },
    dataType: 'json',
    success: function (response) {
      if (response.status === 'success') {
        document.getElementById(cardId)?.remove();
        loadExistingModules();
        refreshModulesUI(course_id);
        alert('Module removed from course_module successfully.');
      }
    },
    error: function () {
      alert('AJAX error occurred while deleting.');
    }
  });
});
</script>







</body>
</html>