<?php
require_once('../includes/connection.php');
header('Content-Type: application/json');

if (!isset($_GET['quiz_id'])) {
    echo json_encode(["status" => "error", "message" => "Quiz ID is required"]);
    exit();
}

$quizId = intval($_GET['quiz_id']);
$courseId = intval($_GET['course_id']);
$user_id = $_SESSION['user_id']; // TODO: replace with session user_id

try {
    // Fetch quiz details first
    $sql = "SELECT * FROM quiz_master WHERE quiz_id = :quiz_id";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':quiz_id', $quizId, PDO::PARAM_INT);
    $stmt->execute();
    $quiz = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$quiz) {
        echo json_encode(["status" => "error", "message" => "Quiz not found."]);
        exit;
    }

    // Fetch latest attempt
    $stmt = $pdo->prepare("
        SELECT *
        FROM quiz_attempts
        WHERE quiz_id = ? AND user_id = ? AND course_id = ?
        ORDER BY quiz_attempt_id DESC
        LIMIT 1
    ");
    $stmt->execute([$quizId, $user_id, $courseId]);
    $attempt = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$attempt) {
        echo json_encode([
            "status"      => "success",
            "quiz_title"  => $quiz['quiz_title'],
            "total_marks" => $quiz['total_marks'],
            "questions"   => []
        ]);
        exit;
    }

    $questionIds = array_filter(array_map('trim', explode(',', $quiz['question_ids'])));
    if (empty($questionIds)) {
        echo json_encode([
            "status"      => "success",
            "quiz_title"  => $quiz['quiz_title'],
            "total_marks" => $quiz['total_marks'],
            "questions"   => []
        ]);
        exit;
    }

    // Fetch questions
    $placeholders = rtrim(str_repeat('?,', count($questionIds)), ',');
    $qSql = "SELECT * FROM question_master WHERE question_master_id IN ($placeholders)";
    $qStmt = $pdo->prepare($qSql);
    $qStmt->execute($questionIds);
    $questions = $qStmt->fetchAll(PDO::FETCH_ASSOC);

    // Fetch selected answers for this attempt
    $qaSql = "SELECT question_id, selected_answer FROM quiz_attempts_data WHERE quiz_attempt_id = ?";
    $qaStmt = $pdo->prepare($qaSql);
    $qaStmt->execute([$attempt['quiz_attempt_id']]);
    $attemptData = $qaStmt->fetchAll(PDO::FETCH_ASSOC);

    // Map answers by question_id
    $selectedAnswersMap = [];
    foreach ($attemptData as $row) {
        $selectedAnswersMap[$row['question_id']] = $row['selected_answer'];
    }

    // Format response
    $formattedQuestions = array_map(function ($q) use ($selectedAnswersMap) {
        $options = [
            'option_1' => $q['option_1'],
            'option_2' => $q['option_2'],
            'option_3' => $q['option_3'],
            'option_4' => $q['option_4'],
        ];

        $type = strtolower(trim($q['type']));
        $correctAnswerValues = [];

        if ($type === 'yes_no' || $type === 'true_false') {
            $val = strtolower(trim($q['correct_answer']));
            if (!empty($val)) {
                $correctAnswerValues[] = ucfirst($val);
            }
        } else {
            $correctAnswerKeys = array_map('trim', explode(',', $q['correct_answer']));
            foreach ($correctAnswerKeys as $key) {
                $keyFormatted = strtolower(str_replace(' ', '_', $key));
                if (isset($options[$keyFormatted])) {
                    $correctAnswerValues[] = $options[$keyFormatted];
                }
            }
        }

        // Fetch selected answer (if available)
        $selectedAnswerRaw = $selectedAnswersMap[$q['question_master_id']] ?? null;
        $selectedAnswerValues = [];

        if ($selectedAnswerRaw) {
            if ($type === 'yes_no' || $type === 'true_false') {
                $selectedAnswerValues[] = ucfirst(strtolower($selectedAnswerRaw));
            } else {
                $keys = array_map('trim', explode(',', $selectedAnswerRaw));
                foreach ($keys as $key) {
                    $keyFormatted = strtolower(str_replace(' ', '_', $key));
                    if (isset($options[$keyFormatted])) {
                        $selectedAnswerValues[] = $options[$keyFormatted];
                    }
                }
            }
        }

        return [
            "question_id"      => $q["question_master_id"],
            "type"             => $q["type"],
            "question"         => $q["question"],
            "media"            => [
                "image"      => $q["image_url"],
                "video"      => $q["video_url"],
                "video_type" => $q["video_type"]
            ],
            "options"          => array_values(array_filter($options)),
            "correct_answer"   => $correctAnswerValues,
            "selected_answer"  => $selectedAnswerValues, // ✅ now included
            "hint"             => $q["hint"],
            "marks"            => $q["marks"]
        ];
    }, $questions);

    echo json_encode([
        "status"      => "attempted",
        "quiz_title"  => $quiz['quiz_title'],
        "total_marks" => $quiz['total_marks'],
        "questions"   => $formattedQuestions
    ]);

} catch (PDOException $e) {
    echo json_encode(["status" => "error", "message" => "Database Error: " . $e->getMessage()]);
}
